# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, division

import os
import tarfile
import logging
import datetime
import textwrap
import time
import email.utils

from sandbox import sdk2, common
from sandbox.sdk2.vcs import svn
from sandbox.common.utils import server_url
from sandbox.common.types import (
    resource as ctr, task as ctt, notification as ctn
)
from sandbox.common.errors import ReleaseError, TaskFailure

from sandbox.projects.common.build import YaPackage
from sandbox.projects.common import constants

from .. import (
    STATBOX_ABT_BASE_LXC_IMAGE,
    STATBOX_ABT_METRICS_BUNDLE,
    STATBOX_ABT_BACKEND_BUNDLE,
    STATBOX_ABT_METRICS_BIN,
)
from ..utils.pyenv import PY2, PY3, PyEnvManager
from ..utils.qloud import QloudApi
from ..utils.task_subprocess import run_task_subprocess

logger = logging.getLogger(__name__)

CHANGELOG_PATH = 'debian/changelog'

ARCADIA_BASE_URL = 'arcadia:/arc/trunk/arcadia'

ARCADIA_METRICS_DIR = 'statbox/abt/metrics'
ARCADIA_METRICS_URL = os.path.join(ARCADIA_BASE_URL, ARCADIA_METRICS_DIR)
ARCADIA_BACKEND_URL = os.path.join(ARCADIA_BASE_URL, 'statbox/abt/backend')

METRICS_DIR = 'statbox-abt-metrics'
METRICS_BUNDLE_DIR = 'statbox-abt-metrics-bundle'
METRICS_BIN_DIR = 'statbox-abt-metrics-bin'
METRICS_BUNDLE_TAR = 'statbox-abt-metrics-bundle.tar.gz'
METRICS_DOCUMENTATION_DIR = 'statbox-abt-metrics-documentation'
METRICS_DOCUMENTATION_TAR = 'statbox-abt-metrics-documentation.tar.gz'
METRICS_PACKAGE_FILE = os.path.join(ARCADIA_METRICS_DIR, 'bin/package.json')

BACKEND_DIR = 'statbox-abt-backend'
BACKEND_BUNDLE_DIR = 'statbox-abt-backend-bundle'
BACKEND_BUNDLE_TAR = 'statbox-abt-backend-bundle.tar.gz'
BACKEND_DOCUMENTATION_DIR = 'statbox-abt-backend-documentation'
BACKEND_DOCUMENTATION_TAR = 'statbox-abt-backend-documentation.tar.gz'

YA_PACKAGE_TIMEOUT = 60 * 60
RELEASE_SUBTASK_TIMEOUT = 10 * 60

TERMINAL_STATUSES = ctt.Status.Group.FINISH + ctt.Status.Group.BREAK

EXECUTE_TERMINAL_STATUSES = (
    ctt.Status.SUCCESS,
    ctt.Status.FAILURE,
    ctt.Status.EXCEPTION,
    ctt.Status.TIMEOUT,
    ctt.Status.STOPPED,
    ctt.Status.EXPIRED,
    ctt.Status.NO_RES,
    ctt.Status.DELETED,
)


def _check_for_status(task_id, expected_status):
    task = sdk2.Task[task_id]

    if task.status != expected_status:
        raise TaskFailure(
            'Subtask {} expected to be in {} status, but have {} status'.format(
                task_id, expected_status, task.status
            )
        )


class StatboxAbtRelease(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 5 * 1024
        ram = 2 * 1024
        privileged = True

        class Caches(sdk2.Requirements.Caches):
            pass

    Requirements = Requirements  # type: StatboxAbtRelease.Requirements

    class Context(sdk2.Task.Context):
        metrics_bundle_id = 0
        backend_bundle_id = 0

        metrics = None
        backend = None

        changelog = ''
        emails = ''

        email_references = ''

        child_tasks_ids = None

    Context = Context  # type: StatboxAbtRelease.Context

    class Parameters(sdk2.Task.Parameters):
        description = 'New statbox-abt release'

        with sdk2.parameters.String('Changelog for metrics') as metrics_changelog_type:
            metrics_changelog_type.values.NONE = metrics_changelog_type.Value('No changelog')
            metrics_changelog_type.values.LATEST_CHANGELOG = metrics_changelog_type.Value('From last revision of debian/changelog', default=True)

        with sdk2.parameters.String('Changelog for backend') as backend_changelog_type:
            backend_changelog_type.values.NONE = backend_changelog_type.Value('No changelog')
            backend_changelog_type.values.LATEST_CHANGELOG = backend_changelog_type.Value('From last revision of debian/changelog', default=True)

        # XXX Когда-нибудь в будущем это будет возвращено

        # with sdk2.parameters.String('Metrics release type') as metrics_release_type:
        #     metrics_release_type.values.HOTFIX = metrics_release_type.Value(default=True)
        #     metrics_release_type.values.MINOR = metrics_release_type.Value()
        #     metrics_release_type.values.MAJOR = metrics_release_type.Value()

        # with sdk2.parameters.String('Changelog for metrics') as metrics_changelog_type:
        #     metrics_changelog_type.values.NONE = metrics_changelog_type.Value('No changelog')
        #     metrics_changelog_type.values.NEW_LATEST = metrics_changelog_type.Value('Between HEAD and revision of debian/changelog', default=True)
        #     metrics_changelog_type.values.HEAD_CUSTOM = metrics_changelog_type.Value('Between HEAD and a custom revision')

        #     with metrics_changelog_type.value.HEAD_CUSTOM:
        #         metrics_custom_tag = sdk2.parameters.String('Custom revision', description='...for metrics package', required=True)

        # with sdk2.parameters.String('Backend release type') as backend_release_type:
        #     backend_release_type.values.HOTFIX = backend_release_type.Value(default=True)
        #     backend_release_type.values.MINOR = backend_release_type.Value()
        #     backend_release_type.values.MAJOR = backend_release_type.Value()

        # with sdk2.parameters.String('Changelog for backend') as backend_changelog_type:
        #     backend_changelog_type.values.NONE = backend_changelog_type.Value('No changelog')
        #     backend_changelog_type.values.NEW_LATEST = backend_changelog_type.Value('Between HEAD and revision if debian/changelog', default=True)
        #     backend_changelog_type.values.HEAD_CUSTOM = backend_changelog_type.Value('Between HEAD and a custom revision')

        #     with backend_changelog_type.value.HEAD_CUSTOM:
        #         backend_custom_tag = sdk2.parameters.String('Custom revision', description='...for backend package', required=True)

        with sdk2.parameters.Group('Svn/Arcadia settings') as access_settings:
            metrics_arcadia_url = sdk2.parameters.String(
                'Metric groups Arcadia url', required=True, default=ARCADIA_METRICS_URL,
                description='Url of the statbox-abt-metrics Arcadia directory'
            )

            backend_arcadia_url = sdk2.parameters.String(
                'Backend Arcadia url', required=True, default=ARCADIA_BACKEND_URL,
                description='Url of the statbox-abt-backend Arcadia directory'
            )

            ssh_key_vault_item_owner = sdk2.parameters.String(
                'SSH key vault item owner',
                default='STATBOX_ABT', required=True,
            )  # type: str

            ssh_key_vault_item_name = sdk2.parameters.String(
                'SSH key vault item name',
                default='ssh_key', required=True,
            )  # type: str

        with sdk2.parameters.Group('Binary build options') as ya_package_options:
            use_aapi_fuse = sdk2.parameters.Bool('Use arcadia-api fuse', default=True)
            run_tests = sdk2.parameters.Bool('Run tests after build', default=True)

        with sdk2.parameters.Group('Qloud deploy settings') as deploy_settings:
            qloud_api_host = sdk2.parameters.Url(
                'Qloud api host',
                default='https://qloud.yandex-team.ru', required=True,
            )  # type: str

            qloud_token_vault_item_owner = sdk2.parameters.String(
                'Qloud token vault item owner',
                default='STATBOX_ABT', required=True,
            )  # type: str

            qloud_token_vault_item_name = sdk2.parameters.String(
                'Qloud token vault item name',
                default='qloud_token', required=True,
            )  # type: str

            qloud_stable = sdk2.parameters.String(
                'Qloud stable main component',
                default='statbox.abt.stable.main',
            )  # type: str

            qloud_testing = sdk2.parameters.String(
                'Qloud testing main component',
                default='statbox.abt.beta.main',
            )  # type: str

        container = sdk2.parameters.Container(
            'LXC Container',
            resource_type=STATBOX_ABT_BASE_LXC_IMAGE,
            required=True,
            register_dependency=False,
        )

    Parameters = Parameters  # type: StatboxAbtRelease.Parameters

    def on_create(self):
        last_container = STATBOX_ABT_BASE_LXC_IMAGE \
            .find(state=ctr.State.READY,
                  attrs={'released': 'stable'}) \
            .order(-STATBOX_ABT_BASE_LXC_IMAGE.id) \
            .first()

        if not last_container:
            last_container = STATBOX_ABT_BASE_LXC_IMAGE \
                .find(state=ctr.State.READY) \
                .order(-STATBOX_ABT_BASE_LXC_IMAGE.id) \
                .first()

        if last_container:
            self.Parameters.container = last_container.id

    def on_finish(self, prev_status, status):
        if not self.Context.child_tasks_ids:
            return

        for task_id in self.Context.child_tasks_ids:
            task = sdk2.Task[task_id]

            if task.status not in TERMINAL_STATUSES:
                task.stop()

    def on_prepare(self):
        if self.Context.metrics:
            logger.info('Skip on_prepare due to already prepared Context')
            return

        logger.info('Preparing context at on_prepare')

        with self.ssh_key():
            for name in ('metrics', 'backend'):
                raw_url = getattr(self.Parameters, '{}_arcadia_url'.format(name))
                url = svn.Arcadia.freeze_url_revision(raw_url)
                changelog_revision = self._get_changelog_revision(url)

                clone_url = svn.Arcadia.replace(url, revision=changelog_revision)
                logger.info('Cloning %s from url: %s', name, clone_url)
                path = svn.Arcadia.get_arcadia_src_dir(clone_url)

                context = {}
                setattr(self.Context, name, context)

                context['url'] = url
                context['clone_url'] = clone_url
                context['path'] = path
                context['changelog_revision'] = changelog_revision
                context['revision'] = svn.Arcadia.info(path)['commit_revision']

                raw_changelog = self._get_changelog_from_log(context)
                context['commiters'] = list({item['author'] for item in raw_changelog})

    def _get_changelog_revision(self, url):
        parsed_url = svn.Arcadia.parse_url(url)
        changelog_url = svn.Arcadia.replace(
            url,
            path=os.path.join(parsed_url.path, CHANGELOG_PATH),
        )
        changelog_info = svn.Arcadia.info(changelog_url)

        if changelog_info:
            changelog_revision = changelog_info['commit_revision']
            logging.info('Extract from url %s changelog revision %s', url, changelog_revision)
            return changelog_revision

        logging.info('Failed to extract changelog revision from url %s', url)
        return None

    def _get_changelog_from_log(self, context):
        # Здесь мы получаем ревизию, в которой ченджлог рпдактировали последний раз.
        # Лучше способа я не придумал.
        url = context['url']
        prev_revision = str(int(context['changelog_revision']) - 1)
        prev_url = svn.Arcadia.replace(url, revision=prev_revision)
        prev_changelog_revision = self._get_changelog_revision(prev_url)
        context['prev_changelog_revision'] = prev_changelog_revision

        if prev_changelog_revision:
            return svn.Arcadia.log(
                url,
                revision_from=prev_changelog_revision,
                revision_to=context['changelog_revision']
            )

        return []

    def on_execute(self):
        with self.memoize_stage.fill_context:
            commiters = set()
            for name in ('metrics', 'backend'):
                context = getattr(self.Context, name)
                changelog_type = getattr(self.Parameters, '{}_changelog_type'.format(name))

                context['version'] = self.get_changelog_field(name, 'version')

                if changelog_type == 'LATEST_CHANGELOG':
                    context['changelog'] = self.get_changelog_field(name, 'changes')
                    commiters.update(context['commiters'])
                elif changelog_type == 'NONE':
                    context['changelog'] = 'statbox-abt-{}: <no changes>'.format(name)

            self.Context.emails = sorted(commiters)
            self.Context.changelog = '\n\n'.join(filter(
                None,
                (self.Context.backend['changelog'], self.Context.metrics['changelog'])
            ))

            self.Context.save()

        with self.memoize_stage.start_ya_package:
            description = 'Build statbox-abt-metrics-bin {}'.format(self.Context.metrics['version'])
            arcadia_url = svn.Arcadia.replace(ARCADIA_BASE_URL, revision=self.Context.metrics['changelog_revision'])
            changelog = os.path.join(ARCADIA_METRICS_DIR, CHANGELOG_PATH)

            YaPackageClass = sdk2.Task['YA_PACKAGE']
            subtask = YaPackageClass(
                self,
                # variables
                description=description,
                checkout_arcadia_from_url=arcadia_url,
                packages=METRICS_PACKAGE_FILE,
                changelog=changelog,
                # parameters
                run_tests=self.Parameters.run_tests,
                use_aapi_fuse=self.Parameters.use_aapi_fuse,
                # constants
                use_new_format=True,
                host_platform='linux',
                package_type=YaPackage.TARBALL,
                build_type=constants.RELEASE_BUILD_TYPE,
                strip_binaries=True,
                resource_type=STATBOX_ABT_METRICS_BIN.name,
            ).enqueue()

            self.Context.child_tasks_ids = [subtask.id]
            self.Context.build_bin_task_id = subtask.id
            self.Context.save()

        with self.memoize_stage.build_python_envs:
            env = PyEnvManager(self)

            common.fs.copy_dir(self.Context.backend['path'], BACKEND_DIR)
            with common.fs.WorkDir(BACKEND_DIR):
                env.python(PY3, 'setup.py', 'write_version',
                           '--version', self.Context.backend['version'])

            self.Context.metrics_bundle_id = self.build_bundle(
                env=env,
                python_version=PY2,
                package_version=self.Context.metrics['version'],
                build_src=self.Context.metrics['path'],
                build_dir=METRICS_BUNDLE_DIR,
                out_file=METRICS_BUNDLE_TAR,
                resource_cls=STATBOX_ABT_METRICS_BUNDLE,
            )

            self.Context.backend_bundle_id = self.build_bundle(
                env=env,
                python_version=PY3,
                package_version=self.Context.backend['version'],
                build_src=BACKEND_DIR,
                build_dir=BACKEND_BUNDLE_DIR,
                out_file=BACKEND_BUNDLE_TAR,
                resource_cls=STATBOX_ABT_BACKEND_BUNDLE,
            )

            self.Context.save()

        with self.memoize_stage.waiting_for_task:
            raise sdk2.WaitTask(
                self.Context.child_tasks_ids,
                EXECUTE_TERMINAL_STATUSES,
                wait_all=True,
                timeout=YA_PACKAGE_TIMEOUT
            )

        # Ниже этой точки, я не уверен, присутствует ли еще копия репозитория
        # на диске.
        # Дело в том, что после ожидания, повторно вызывается on_prepare.
        # Но там я поставил заглушку, чтобы на повторный вызов он ничего не делал,
        # в т.ч. и не скачивал бы Аркадию.

        with self.memoize_stage.finising:
            _check_for_status(self.Context.build_bin_task_id, ctt.Status.SUCCESS)

            build_bin_task = sdk2.Task[self.Context.build_bin_task_id]

            bin_resource = sdk2.Resource[STATBOX_ABT_METRICS_BIN.name].find(
                task=build_bin_task
            ).first()

            self.Context.metrics_bin_resource_id = bin_resource.id
            self.Context.metrics_bin_resource_filename = str(bin_resource.path)

    def get_changelog_field(self, package_name, field):
        package_dir = {
            'metrics': self.Context.metrics['path'],
            'backend': self.Context.backend['path'],
        }[package_name]

        logger.info(
            'Extracting field `%s` from changelog %s of package `%s`...',
            field, os.path.join(package_dir, CHANGELOG_PATH), package_name
        )

        value = run_task_subprocess(
            self,
            ['dpkg-parsechangelog', '-S', field],
            cwd=package_dir,
            read_stdout=True
        ).strip()

        logger.info('Extracted field `%s`:\n%s', field, value)

        return value

    def build_bundle(
        self,
        env,
        python_version,
        package_version,
        build_src,
        build_dir,
        out_file,
        resource_cls,
        additional_paths=(),
    ):
        with env.virtualenv(python_version) as venv:
            logger.info('created new virtualenv at %s', venv._tmp_dir)
            logger.info('install package to virtualenv from %s', build_src)

            venv.install(os.path.abspath(build_src), '-v')
            venv.make_relocatable()

            with tarfile.open(out_file, 'w:gz') as t:
                t.add(venv.basedir, build_dir)
                for path in additional_paths:
                    t.add(*path)

        desc = 'Relocatable virtualenv for {}'.format(build_src)
        resource = resource_cls(self, desc, out_file,
                                python_version=python_version,
                                package_version=package_version)

        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()

        return resource.id

    def ssh_key(self):
        return sdk2.ssh.Key(
            self,
            self.Parameters.ssh_key_vault_item_owner,
            self.Parameters.ssh_key_vault_item_name
        )

    def on_release(self, parameters):
        status = parameters['release_status']
        build_bin_task_id = self.Context.build_bin_task_id

        self.send_release_start_email(parameters)

        try:
            build_bin_task = sdk2.Task[build_bin_task_id]
            build_bin_task.mark_released_resources(status)
        except Exception as e:
            text = 'Resources of subtask YA_PACKAGE {} is not released\n\n{}'.format(build_bin_task_id, e)
            self.send_release_finish_email(parameters, text)
            raise
        else:
            self.set_info('Resources of subtask YA_PACKAGE {} is released'.format(build_bin_task_id))

        if status == 'stable':
            qloud_path = self.Parameters.qloud_stable
        elif status == 'testing':
            qloud_path = self.Parameters.qloud_testing
        else:
            raise ReleaseError('can only deploy to stable/testing')

        if not qloud_path:
            raise ReleaseError('{} main component is not specified'
                               .format(status))

        # reset email thread so that emails from different releases
        # appear in different threads.
        self.Context.email_references = ''

        try:
            self.do_on_release(parameters, qloud_path)
        except Exception as e:
            text = 'Environment is not deployed\n\n' + str(e)
            self.send_release_finish_email(parameters, text)
            raise
        else:
            text = 'Environment successfully deployed'
            self.set_info(text)
            self.send_release_finish_email(parameters, text)

    def do_on_release(self, parameters, qloud_path):
        environment, component_name = qloud_path.rsplit('.', 1)

        self.mark_released_resources(parameters['release_status'])

        token = sdk2.Vault.data(
            self.Parameters.qloud_token_vault_item_owner,
            self.Parameters.qloud_token_vault_item_name
        )

        qloud_api = QloudApi(token, self.Parameters.qloud_api_host)

        data = qloud_api.dump_environment(environment)

        data['comment'] = 'Release m={}/b={} to {} by {} (task={})'.format(
            str(self.Context.metrics['version']),
            str(self.Context.backend['version']),
            parameters['release_status'],
            parameters['releaser'],
            self.id
        )

        component = self.find_component(data, component_name)

        environment_variables = component['environmentVariables']

        bundle_id = str(self.Context.metrics_bundle_id)
        environment_variables['SANDBOX_TASK_WHEEL_BUNDLE'] = bundle_id

        metrics_bin_resource_id = str(self.Context.metrics_bin_resource_id)
        environment_variables['SANDBOX_METRICS_BIN_RESOURCE_ID'] = metrics_bin_resource_id

        metrics_version = str(self.Context.metrics['version'])
        environment_variables['METRICS_VERSION'] = metrics_version

        backend_version = str(self.Context.backend['version'])
        environment_variables['BACKEND_VERSION'] = backend_version

        component['sandboxResources'] = [
            {
                'id': self.Context.metrics_bundle_id,
                'localName': METRICS_BUNDLE_TAR,
                'symlink': '/' + METRICS_BUNDLE_DIR,
                'dynamic': False,
                'extract': True,
            },
            {
                'id': self.Context.backend_bundle_id,
                'localName': BACKEND_BUNDLE_TAR,
                'symlink': '/' + BACKEND_BUNDLE_DIR,
                'dynamic': False,
                'extract': True,
            },
            {
                'id': self.Context.metrics_bin_resource_id,
                'localName': self.Context.metrics_bin_resource_filename,
                'symlink': '/' + METRICS_BIN_DIR,
                'dynamic': False,
                'extract': True,
            }
        ]

        deploy_started = datetime.datetime.utcnow()
        deploy_timeout = datetime.timedelta(minutes=30)

        qloud_api.upload_environment(data)

        while True:
            status = qloud_api.get_environment_status(environment)

            if status['status'] == 'DEPLOYED':
                return
            elif status['status'] == 'FAILED':
                raise ReleaseError(status['statusMessage'])
            elif datetime.datetime.utcnow() - deploy_started > deploy_timeout:
                raise ReleaseError('Deployment hangs for too long, '
                                   'manual check required')

            time.sleep(1)

    @staticmethod
    def find_component(data, name):
        for component in data['components']:
            if component['componentName'] == name:
                return component
        else:
            raise RuntimeError('cannot find component {}'.format(name))

    def send_release_start_email(self, parameters):
        releaser = parameters['releaser']
        release_status = parameters['release_status']
        message_subject = parameters.get('release_subject')
        release_to = parameters['email_notifications']['to']
        release_cc = parameters['email_notifications']['cc']

        if not release_to:
            release_to, release_cc = release_cc, release_to
            if not release_to:
                logging.info('No recipients specified for the release.')
                return

        mail_subj = '[{}] {}'.format(release_status, message_subject)

        text = textwrap.dedent('''
            <b>Releaser:</b> {releaser}
            <b>Release status:</b> {release_status}
            <b>Metrics package version:</b> {metrics_version}
            <b>Backend package version:</b> {backend_version}
            <b>Sandbox task:</b> {host}/task/{task_id}/view
        ''').lstrip().format(
            release_status=release_status,
            releaser=releaser,
            metrics_version=self.Context.metrics['version'],
            backend_version=self.Context.backend['version'],
            host=server_url(),
            task_id=self.id,
        )

        change_log = parameters.get('release_comments')

        if change_log:
            text += '\n\n' + change_log.strip() + '\n'

        text += '\n\n--\nstatbox-abt team\n'

        self.send_email(
            subject=mail_subj,
            body=text.replace('\n', '<br>\n'),
            recipients=release_to,
            transport=ctn.Transport.EMAIL,
            type=ctn.Type.HTML,
            charset=ctn.Charset.UTF,
            task_id=self.id,
            view=ctn.View.DEFAULT
        )

    def send_release_finish_email(self, parameters, text):
        release_status = parameters['release_status']
        message_subject = parameters.get('release_subject')
        release_to = parameters['email_notifications']['to']
        release_cc = parameters['email_notifications']['cc']

        if not release_to:
            release_to, release_cc = release_cc, release_to
            if not release_to:
                logging.info('No recipients specified for the release.')
                return

        text += '\n\n--\nstatbox-abt team\n'

        mail_subj = '[{}] {}'.format(release_status, message_subject)

        # for some mysterious reason, apple mail agent fails to group messages
        # together if they are of different content types...
        text = text.replace('&', '&amp;')
        text = text.replace('<', '&lt;')
        text = text.replace('>', '&gt;')
        text = text.replace('\n', '<br>\n')

        self.send_email(
            subject=mail_subj,
            body=text,
            recipients=release_to,
            transport=ctn.Transport.EMAIL,
            type=ctn.Type.HTML,
            charset=ctn.Charset.UTF,
            task_id=self.id,
            view=ctn.View.DEFAULT
        )

    def send_email(self, **kwargs):
        for i in range(len(kwargs['recipients'])):
            recipient = kwargs['recipients'][i]
            if recipient.endswith('@yandex-team.ru'):
                kwargs['recipients'][i] = recipient[:-len('@yandex-team.ru')]

        msg_id = email.utils.make_msgid()
        headers = ['Message-Id: ' + msg_id]

        if self.Context.email_references:
            in_reply_to = self.Context.email_references.split()[-1]
            headers.append('In-Reply-To: ' + in_reply_to)
            headers.append('References: ' + self.Context.email_references)

        self.Context.email_references += ' ' + msg_id

        kwargs['headers'] = headers

        logger.info('sending email %r', kwargs)

        # noinspection PyBroadException
        try:
            self.server.notification(**kwargs)
        except Exception:
            logging.exception('')

    @property
    def release_template(self):
        recipients = {'lipkin'}

        if self.Context.emails:
            recipients.update(self.Context.emails)

        return sdk2.ReleaseTemplate(
            sorted(recipients),
            'Release statbox-abt metrics={}/backend={}'.format(
                str(self.Context.metrics.get('version', 'unknown')),
                str(self.Context.backend.get('version', 'unknown')),
            ),
            self.Context.changelog.strip(),
            [ctt.ReleaseStatus.TESTING, ctt.ReleaseStatus.STABLE]
        )
