# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, division

import contextlib
import logging
import os
import shutil
import tempfile
import weakref

from sandbox import sdk2

PY2 = '2.7.13'
PY3 = '3.6.2'
PYTHON_VERSIONS = (PY2, PY3)

PYENV_ROOT = '/pyenv'
PYENV_REPO = 'https://github.com/pyenv/pyenv.git'

PYPI = 'https://pypi.yandex-team.ru/simple'


class PyEnvManager(object):
    def __init__(self, task,
                 python_versions=PYTHON_VERSIONS,
                 pyenv_basedir=PYENV_ROOT):

        self._python_versions = python_versions
        self._pyenv_basedir = pyenv_basedir
        self._task = weakref.ref(task)

        self._logger = logging.getLogger('pyenv_manager')

        self._init_default_python(PY2)

    @contextlib.contextmanager
    def _redirect_log(self):
        task = self._task()
        if task is None:
            yield None, None
            return
        else:
            with sdk2.helpers.ProcessLog(task, self._logger) as process_log:
                yield process_log.stdout, process_log.stderr

    def execute(self, *args):
        self._logger.info('running %s', ' '.join(args))
        with self._redirect_log() as (stdout, stderr):
            process = sdk2.helpers.subprocess.Popen(args, stdout=stdout, stderr=stderr)
            exitcode = process.wait()

        if exitcode != 0:
            raise RuntimeError('command {} exited with non-zero exitcode {}'
                               .format(' '.join(args), exitcode))

    def my_dir(self, version):
        if version == 'system':
            path = '/usr/'
        elif version in self._python_versions:
            path = os.path.join(self._pyenv_basedir, 'versions', version)
        else:
            raise RuntimeError('unknown python version {}'.format(version))

        if not os.path.exists(path):
            raise RuntimeError('failed to find dir for {}; {} is does not exists'.format(version, path))

        if not os.path.isdir(path):
            raise RuntimeError('failed to find dir for {}; {} is not a directory'.format(version, path))

        return path

    def my_bin_dir(self, version):
        path = os.path.join(self.my_dir(version), 'bin')

        if not os.path.exists(path):
            raise RuntimeError('failed to find bin dir for {}; {} is does not exists'.format(version, path))

        if not os.path.isdir(path):
            raise RuntimeError('failed to find bin dir for {}; {} is not a directory'.format(version, path))

        return path

    def _which(self, version, binary):
        path = os.path.join(self.my_bin_dir(version), binary)

        if not os.path.exists(path):
            raise RuntimeError('failed to find binary for {}={}; {} is does not exists'.format(binary, version, path))

        return path

    def which_python(self, version):
        return self._which(version, 'python')

    def which_pip(self, version):
        return self._which(version, 'pip')

    def python(self, version, *args):
        self.execute(self.which_python(version), *args)

    def pip(self, version, *args):
        self.execute(self.which_pip(version), *args)

    def install(self, version, *packages):
        self.pip(version, 'install', '-i', PYPI, '-U', *packages)

    def wheel(self, version, what, where):
        self.pip(version, 'wheel', '-i', PYPI, what, '-w', where)

    def virtualenv(self, version):
        return VirtualEnvManager(self, version)

    def _init_default_python(self, version):
        my_bin_dir = self.my_bin_dir(version)

        if not os.path.exists(os.path.join(my_bin_dir, 'python')):
            raise RuntimeError('cannot set default python to {}: there is no python executable in {}'.format(version, my_bin_dir))

        if not os.path.exists(os.path.join(my_bin_dir, 'pip')):
            raise RuntimeError('cannot set default python to {}: there is no pip executable in {}'.format(version, my_bin_dir))

        if not os.path.exists(os.path.join(my_bin_dir, 'virtualenv')):
            raise RuntimeError('cannot set default python to {}: there is no virtualenv executable in {}'.format(version, my_bin_dir))

        os.environ['PATH'] = my_bin_dir + ':' + os.environ['PATH']


class VirtualEnvManager(object):
    def __init__(self, pyenv_manager, python_version):
        self._python_version = python_version
        self._pyenv_manager = pyenv_manager

        self._tmp_dir = None
        self._environ = None

    def __enter__(self):
        if self._tmp_dir is not None:
            raise RuntimeError('{} is not reentrant'.format(self.__class__.__name__))

        self._tmp_dir = tempfile.mkdtemp()
        self._environ = os.environ.copy()

        _path = self._pyenv_manager.which_python(self._python_version)
        self._pyenv_manager.python(self._python_version, '-m', 'virtualenv', self._tmp_dir, '-p', _path)

        my_bin_dir = os.path.join(self._tmp_dir, 'bin')

        os.environ['PATH'] = my_bin_dir + ':' + os.environ['PATH']
        os.environ['VIRTUAL_ENV'] = self._tmp_dir

        self.install('pip', 'setuptools', 'wheel')

        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        shutil.rmtree(self._tmp_dir, ignore_errors=True)

        for key in set(self._environ.keys() + os.environ.keys()):
            if key not in self._environ:
                os.environ.pop(key)
            else:
                os.environ[key] = self._environ[key]

        self._path = None
        self._tmp_dir = None

    @property
    def basedir(self):
        return self._tmp_dir

    def execute(self, *args):
        self._pyenv_manager.execute(*args)

    def _which(self, binary):
        if self._tmp_dir is None:
            raise RuntimeError('this virtualenv is not initialised')

        path = os.path.join(self._tmp_dir, 'bin', binary)

        if not os.path.exists(path):
            raise RuntimeError('failed to find binary for {}; {} is empty'.format(binary, path))

        return path

    def which_python(self):
        return self._which('python')

    def which_pip(self):
        return self._which('pip')

    def python(self, *args):
        self.execute(self.which_python(), *args)

    def pip(self, *args):
        self.execute(self.which_pip(), *args)

    def install(self, *packages):
        self.pip('install', '-i', PYPI, '-U', *packages)

    def wheel(self, what, where):
        self.pip('wheel', '-i', PYPI, what, '-w', where)

    def make_relocatable(self):
        self._pyenv_manager.python(self._python_version, '-m', 'virtualenv', '--relocatable', self.basedir)
