# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, division

import logging
import requests


logger = logging.getLogger(__name__)
logger.addHandler(logging.StreamHandler())
logger.setLevel(logging.DEBUG)


class QloudApi(object):
    def __init__(self, token, url):
        self._url = url
        self._token = token
        self._headers = {
            'Authorization': 'OAuth {}'.format(self._token),
        }

    def _fetch(self, url, method, params=None, data=None, json=None):
        url = self._url + '/' + url.lstrip('/')

        logger.info('fetching %s %s', method.upper(), url)

        methods = {
            'GET': requests.get,
            'POST': requests.post,
            'PUT': requests.put,
            'DELETE': requests.delete,
        }

        method = methods[method.upper()]

        response = method(
            url,
            params=params,
            data=data,
            json=json,
            headers=self._headers
        )

        logger.info('got response %s', response.status_code)

        if response.status_code != 200:
            logger.error('Gon non-200 code, error message is %r', response.text)

        response.raise_for_status()

        try:
            return response.json()
        except ValueError:
            return response.text

    def create_environment(self, application_id, name):
        url = '/api/v1/environment/new'
        data = {'applicationId': application_id, 'environmentName': name}
        return self._fetch(url, 'POST', data=data)

    def create_environment_draft(self, id_):
        url = '/api/v1/environment/fork'
        data = {'originId': id_, 'environmentName': id_.rsplit('.', 1)[1]}
        return self._fetch(url, 'POST', data=data)

    def get_environment(self, id_):
        url = '/api/v1/environment/stable/{}'.format(id_)
        return self._fetch(url, 'GET')

    def get_environment_by_version(self, id_, version):
        url = '/api/v1/environment/stable/{}/{}'.format(id_, version)
        return self._fetch(url, 'GET')

    def get_environment_draft(self, id_):
        url = '/api/v1/environment/sandbox/{}'.format(id_)
        return self._fetch(url, 'GET')

    def get_environment_status(self, id_):
        url = '/api/v1/environment/status/{}'.format(id_)
        return self._fetch(url, 'GET')

    def get_environment_status_by_version(self, id_, version):
        url = '/api/v1/environment/status/{}/{}'.format(id_, version)
        return self._fetch(url, 'GET')

    def dump_environment(self, id_):
        url = '/api/v1/environment/dump/{}'.format(id_)
        return self._fetch(url, 'GET')

    def upload_environment(self, data):
        url = '/api/v1/environment/upload'
        return self._fetch(url, 'POST', json=data)

    def deploy_environment_draft(self, id_):
        url = '/api/v1/environment/deploy'
        data = {'environmentId': id_}
        return self._fetch(url, 'POST', data=data)

    def delete_environment(self, id_):
        url = '/api/v1/environment/stable/{}'.format(id_)
        return self._fetch(url, 'DELETE')

    def delete_environment_draft(self, id_):
        url = '/api/v1/environment/sandbox/{}'.format(id_)
        return self._fetch(url, 'DELETE')

    def update_environment_env(self, id_, data):
        url = '/api/v1/environment/env/{}'.format(id_)
        return self._fetch(url, 'POST', data=data)

    def create_component(self, data):
        url = '/api/v1/component'
        return self._fetch(url, 'PUT', data=data)

    def get_component(self, id_):
        url = '/api/v1/component/{}'.format(id_)
        return self._fetch(url, 'GET')

    def modify_component(self, id_, data):
        url = '/api/v1/component/{}'.format(id_)
        return self._fetch(url, 'POST', json=data)

    def get_component_by_version(self, id_, version):
        url = '/api/v1/runtime/{}/{}'.format(id_, version)
        return self._fetch(url, 'GET')

    def delete_component(self, id_):
        url = '/api/v1/component/{}'.format(id_)
        return self._fetch(url, 'DELETE')

    def modify_component_by_version(self, id_, version, data):
        url = '/api/v1/component/{}/{}/deploy'.format(id_, version)
        return self._fetch(url, 'POST', json=data)
