# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging

import sandbox.projects.common.utils2 as pcu2

from sandbox.common import errors
from sandbox.sdk2.helpers import subprocess, ProcessLog


class SubprocessError(errors.TaskError):
    def __init__(self, message, log_resource_id, stderr_path, stdout_path):
        super(SubprocessError, self).__init__(message)

        self._log_resource_id = log_resource_id
        self._stderr_path = stderr_path
        self._stdout_path = stdout_path

    def _get_html_view(self, path, type_):
        filename = path.name

        if not path.is_file() or path.stat().st_size == 0:
            link = '&ltEMPTY&gt'  # <EMPTY>
        else:
            link = pcu2.resource_redirect_link(
                self._log_resource_id, filename, filename
            )

        return '<b>{}:</b> {}'.format(type_, link)

    def get_task_info(self):
        return (
            'Failed subprocess logs:<br>'
            '- {stderr_view}<br>'
            '- {stdout_view}'
            .format(
                stderr_view=self._get_html_view(self._stderr_path, 'stderr'),
                stdout_view=self._get_html_view(self._stdout_path, 'stdout'),
            )
        )


def run_task_subprocess(task, cmd, cwd=None, read_stdout=False):
    logging.info('Start subprocess for task %r', task)
    logging.info(' - command: %r', cmd)
    logging.info(' - working directory: %s', cwd)
    logging.info(' - stdout flag: %r', read_stdout)

    result = None

    logger = cmd[0]

    with ProcessLog(task, logger=logger) as log:
        process = subprocess.Popen(
            cmd,
            cwd=cwd,
            stdout=subprocess.PIPE if read_stdout else log.stdout,
            stderr=log.stderr,
        )

        if read_stdout:
            result, _ = process.communicate()
        else:
            process.wait()

        stderr_path = log.stderr.path
        stdout_path = log.stdout.path

    returncode = process.returncode

    if returncode != 0:
        raise SubprocessError(
            'command {!r} died with exit code {}'.format(cmd, returncode),
            task.log_resource.id,
            stderr_path,
            stdout_path,
        )

    return result
