"""
Base image build: ubuntu xenial,

```
set -e

apt-get -y update
apt-get -y upgrade
apt-get -y install \
    python2.7 python2.7-dev python-pip python-virtualenv \
    python3.5 python3.5-dev python3-pip python3.5-venv \
    build-essential libc6-dev curl \
    git libyaml-dev

mkdir -p /venvs

python2.7 -m virtualenv --never-download /venvs/py2
/venvs/py2/bin/python -V
/venvs/py2/bin/python -m pip -V
/venvs/py2/bin/python -m pip install \
    --index-url https://pypi.yandex-team.ru/simple \
    --disable-pip-version-check \
    --upgrade \
    pip==19.3.1 setuptools==44.1.1 wheel==0.36.2 \
    GitPython==2.1.15 pyyaml==5.4.1 virtualenv==16.7.10
rm -rf ~/.cache/pip

python3.5 -m venv /venvs/py3
/venvs/py3/bin/python -V
/venvs/py3/bin/python -m pip -V
/venvs/py3/bin/python -m pip install \
    --index-url https://pypi.yandex-team.ru/simple \
    --disable-pip-version-check \
    --upgrade \
    pip==20.3.4 setuptools==50.3.2 wheel==0.36.2
rm -rf ~/.cache/pip
```
"""

import os
import json
import logging
import tarfile
import time

from os.path import join, exists, isdir

from subprocess import check_call

import sandbox.common.types.resource as ctr
import sandbox.projects.statinfra

from sandbox.projects.common import link_builder as lb
from sandbox.common.auth import OAuth
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox import sdk2
from sandbox.common import rest
from . import config
from ..extra import juggler_push, STATBOX_PACKAGES_LIST

STATINFRA_ACTION_REPOS = getattr(sandbox.projects.statinfra, config.ACTION_REPOS_RESOURCE_NAME)
STATINFRA_JOB_REPOS = getattr(sandbox.projects.statinfra, config.JOB_REPOS_RESOURCE_NAME)
STATINFRA_LIB_REPOS = getattr(sandbox.projects.statinfra, config.LIB_REPOS_RESOURCE_NAME)
STATINFRA_CONFIG_REPOS = getattr(sandbox.projects.statinfra, config.CONFIG_REPOS_RESOURCE_NAME)

ACTIONS_TAR_NAME = 'actions.tar.gz'
JOBS_TAR_NAME = 'jobs.tar.gz'
LIBS_TAR_NAME = 'libs.tar.gz'
CONFIGS_TAR_NAME = 'configs.tar.gz'

LAST_RESOURCE_ROOT = 'last_resource'
ROOT_PREFIX = '_ROOT_'
REPO_CONFIG_PREFIX = '_REPO_CONFIG_'
SITE_PACKAGES = '_site_packages_'
VENV_PATH = '_VIRTUAL_ENVIRONMENTS_'
STATINFRA_REVS_JSON = '.statinfra_revs.json'
REVISION_PREFIX = 'revision__'


class Retrier(object):
    def __init__(self, num_retries=5, initial_timeout=3, mult_coeff=2):
        self.num_retries = num_retries
        self.initial_timeout = initial_timeout
        self.mult_coeff = mult_coeff

    def run(self, f, *args, **kwargs):
        sleep_time = self.initial_timeout
        for i in range(self.num_retries):
            try:
                return f(*args, **kwargs)
            except Exception:
                logging.exception('Retrying')
                i += 1
                if i >= self.num_retries:
                    raise
                time.sleep(sleep_time)
                sleep_time *= self.mult_coeff


class RepoInfo(object):
    def __init__(self, path, prefix=None):
        self.path = path
        self.prefix = prefix
        self.sha = self._sha()

    def __eq__(self, other):
        return self.sha == other.sha

    def __ne__(self, other):
        return self.sha != other.sha

    def get_last(self):
        return self.__class__(join(LAST_RESOURCE_ROOT, self.prefix), self.path)


class GitInfo(RepoInfo):
    def _sha(self):
        if exists(join(self.path, STATINFRA_REVS_JSON)):
            revs = json.load(open(join(self.path, STATINFRA_REVS_JSON)))
            repo = '{}{}'.format(REVISION_PREFIX, self.prefix)
            return int(revs[repo]) if revs.get(repo, None) else None
        import git
        return git.Repo(self.path).rev_parse('HEAD').hexsha if exists(join(self.path, '.git')) else None


class SvnInfo(RepoInfo):
    def _sha(self):
        if exists(join(self.path, STATINFRA_REVS_JSON)):
            revs = json.load(open(join(self.path, STATINFRA_REVS_JSON)))
            repo = '{}{}'.format(REVISION_PREFIX, self.prefix)
            return int(revs[repo]) if revs.get(repo, None) else None
        return int(svn.Arcadia.info(self.path)['commit_revision']) if exists(join(self.path, '.svn')) else None


class BaseRepo(object):
    def __init__(self, url, _prefix, retrier):
        self.url = url
        if url.endswith('.git'):
            url = url[:-4]
        self.path = '/'.join(url.split('/')[-3:])
        self._prefix = _prefix
        self.retrier = retrier

    def _set_prefix(self):
        if self._prefix == REPO_CONFIG_PREFIX:
            import yaml
            repo_config_path = join(self.path, '.repo_config.tyaml')
            if exists(repo_config_path):
                repo_config = yaml.load(open(repo_config_path))
                for e in ['default', '_default']:
                    if repo_config.get(e):
                        self.prefix = repo_config[e]['templates']['job_prefix']
                        break
            else:
                self.prefix = self.path.split('/')[-1]
        elif self._prefix == ROOT_PREFIX:
            self.prefix = ''
        elif self._prefix:
            self.prefix = self._prefix
        else:
            self.prefix = self.path.split('/')[-1]


def pip_install(*paths):
    base_python = ['/venvs/py2/bin/python']
    python = [join(VENV_PATH, 'bin/python')]
    pip = python + ['-m', 'pip']

    proc_env = os.environ.copy()
    proc_env.pop('PYTHONPATH', None)

    with sdk2.helpers.ProcessLog(logger='pip_install_libs') as proc:
        if not exists(VENV_PATH):
            check_call(
                (
                    base_python +
                    ['-m', 'virtualenv'] +
                    ['--never-download'] +
                    # ['--python', '/usr/bin/python'] +
                    [VENV_PATH]
                ),
                env=proc_env, stdout=proc.stdout, stderr=proc.stderr)

        # Debuggability:
        check_call(python + ['-V'])
        check_call(pip + ['-V'])

        pip_install_cmd = pip + [
            'install',
            '--index-url', 'https://pypi.yandex-team.ru/simple',
            '--disable-pip-version-check',
        ]
        check_call(
            (
                pip_install_cmd +
                ['--upgrade'] +
                ['pip==19.3.1', 'setuptools==44.1.1', 'wheel==0.36.2']
            ),
            env=proc_env, stdout=proc.stdout, stderr=proc.stderr)
        check_call(pip + ['-V'])
        check_call(
            (
                pip_install_cmd +
                ['--target', SITE_PACKAGES] +
                list(paths)
            ),
            env=proc_env, stdout=proc.stdout, stderr=proc.stderr)


class GitRepo(BaseRepo):
    def clone(self):
        import git
        self.retrier.run(git.Git().clone, self.url, self.path)
        self._set_prefix()
        return GitInfo(self.path, self.prefix)


class SvnRepo(BaseRepo):
    def clone(self):
        self.retrier.run(svn.Arcadia.checkout, self.url, self.path)
        self._set_prefix()
        return SvnInfo(self.path, self.prefix)


def repo_factory(repo_url, *args, **kwargs):
    if repo_url.startswith('arcadia:'):
        return SvnRepo(repo_url, *args, **kwargs)
    else:
        return GitRepo(repo_url, *args, **kwargs)


class CreateStatinfraReposResource(sdk2.Task):
    # BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WA
    #
    # This task runs diffirent ways in testing/production.
    # https://st.yandex-team.ru/STATINFRA-10499
    #
    # BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WARNING BIG WA
    name = config.TASK_TYPE

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 20000
        environments = (
            PipEnvironment('GitPython==2.1.15'),
        )

    class Parameters(sdk2.Task.Parameters):
        update_venvs = sdk2.parameters.Bool('Force update resources', default=False)
        # TODO: replace ROOT_PREFIX in lib_repos after disabling old libs making mode
        lib_repos = sdk2.parameters.Dict('Libs repositories (repo: prefix)', required=True)
        actions_repos = sdk2.parameters.Dict('Actions repositories (repo: prefix)', required=True)
        job_repos = sdk2.parameters.Dict('Jobs repositories (repo: prefix)', required=True)
        config_repos = sdk2.parameters.Dict('Configs repositories (repo: prefix)', required=True)

        Container = sdk2.parameters.Container(
            'LXC Container',
            # DMP-53: https://www-sandbox1.n.yandex-team.ru/task/1032078215/view
            # https://sandbox.yandex-team.ru/task/928998656/view
            default=1036129736,
        )

    def on_execute(self):

        retrier = Retrier()

        cases = [
            (STATINFRA_LIB_REPOS, LIBS_TAR_NAME, self.Parameters.lib_repos),
            (STATINFRA_ACTION_REPOS, ACTIONS_TAR_NAME, self.Parameters.actions_repos),
            (STATINFRA_JOB_REPOS, JOBS_TAR_NAME, self.Parameters.job_repos),
            (STATINFRA_CONFIG_REPOS, CONFIGS_TAR_NAME, self.Parameters.config_repos),
        ]

        with ssh.Key(self, 'STATINFRA', 'robot-statinfra-ssh-key'):
            for resource_type, tar_name, repos_list in cases:

                last_resource = resource_type.find(state=ctr.State.READY).first()
                if last_resource is not None:
                    try:
                        tarfile.open(str(sdk2.ResourceData(last_resource).path)).extractall(LAST_RESOURCE_ROOT)
                    except IOError:
                        pass

                repos = list()
                statinfra_revs = dict()
                need_update = self.Parameters.update_venvs

                for repo_address, prefix in repos_list.items():
                    logging.info('--------- REPO ADDRESS %s', repo_address)
                    r = repo_factory(repo_address, prefix, retrier)
                    repo = r.clone()
                    repos.append(repo)
                    old_repo = repo.get_last()
                    if need_update:
                        logging.info('{} needs update'.format(tar_name))
                    else:
                        if repo != old_repo:
                            need_update = True
                            logging.info('{} needs update'.format(tar_name))
                    logging.info('Repo {} (old_sha = {}, sha = {}, path = {}) will pack into {}'.format(
                        repo_address, old_repo.sha, repo.sha, repo.path, repo.prefix)
                    )

                if need_update:
                    tar = tarfile.open(tar_name, 'w:gz')

                    for repo in repos:
                        statinfra_revs['{}{}'.format(REVISION_PREFIX, repo.path)] = repo.sha

                    if resource_type is STATINFRA_LIB_REPOS:
                        pip_install(*[repo.path for repo in repos])

                        if isdir(SITE_PACKAGES):
                            for i in os.listdir(SITE_PACKAGES):
                                tar.add(join(SITE_PACKAGES, i), i)
                    else:
                        for repo in repos:
                            tar.add(repo.path, repo.prefix)

                    with open(STATINFRA_REVS_JSON, 'w') as f:
                        f.write(json.dumps(statinfra_revs))
                    tar.add(STATINFRA_REVS_JSON)
                    statbox_packages_list = '\n'.join(['{} {}'.format(k, v) for k, v in statinfra_revs.iteritems()])
                    with open(STATBOX_PACKAGES_LIST, 'w') as f:
                        f.write(statbox_packages_list)
                    tar.add(STATBOX_PACKAGES_LIST)
                    tar.close()

                    self.Parameters.tags += [tar_name]
                    self.set_info('Resource {} is going to be updated\n{}'.format(resource_type, statbox_packages_list))
                    statinfra_revs['arch'], statinfra_revs['ttl'] = 'any', 60
                    resource = sdk2.ResourceData(resource_type(self, tar_name, tar_name, **statinfra_revs))
                    resource.ready()
                    self.set_info('Resource has been successfully uploaded')

                    if resource_type is STATINFRA_LIB_REPOS:
                        self.set_info('Starting building porto layer with libs')
                        rest_cli = rest.Client(auth=OAuth(sdk2.Vault.data('STATINFRA', 'ROBOT_STATINFRA_TOKEN')))
                        # parent_layer = rest_cli.resource.read(
                        #     type='PORTO_LAYER_YT', state=ctr.State.READY, limit=1,
                        # attrs={"name_without_datetime": "porto_layer_search_ubuntu_precise_app"}
                        # )
                        build_params = {
                            # Resource hardcoded from DMP-34
                            'parent_layer': 2204354453,   # parent_layer['items'][0]['id'],
                            'layer_filename': 'porto_layer_libs',
                            'debug_build': False
                        }
                        build_params.update(config.PORTO_LIBS_BUILD_PARAMS)
                        porto_task = rest_cli.task(dict(
                            type='STATBOX_PORTO_PORNO',
                            owner=self.owner,
                            description='Statinfra libs porto layer ({})'.format(build_params['clusters']),
                            custom_fields=[dict(name=n, value=v) for n, v in build_params.iteritems()]
                        ))
                        rest_cli.batch.tasks.start.update(porto_task['id'])

                        self.set_info('Porto build: {}'.format(lb.task_link(porto_task['id'])), do_escape=False)
                else:
                    self.set_info("Nothing has changed, resource {} ain't gonna be updated".format(resource_type))

    def on_success(self, prev_status):
        juggler_push(
            host=config.JUGGLER_HOST,
            service=config.TASK_TYPE,
            status='OK',
            description='create_statinfra_repos_resource success',
        )
        super(CreateStatinfraReposResource, self).on_success(prev_status)

    def on_failure(self, prev_status):
        juggler_push(
            host=config.JUGGLER_HOST,
            service=config.TASK_TYPE,
            status='CRIT',
            description='create_statinfra_repos_resource failure',
        )
        super(CreateStatinfraReposResource, self).on_failure(prev_status)
