# -*- coding: utf-8 -*-
import os
import logging
import requests
import time
from functools import wraps


VENVS_ROOT = '/venv'
STATBOX_PACKAGES_LIST = 'STATBOX_PACKAGES.LIST'


class chdir(object):
    def __init__(self, dir):
        self.pwd = os.getcwd()
        self.dir = dir

    def __enter__(self):
        os.chdir(self.dir)

    def __exit__(self, *args):
        os.chdir(self.pwd)


def juggler_push(host, service, status, description, tags=[]):
    def requests_retry():
        s = requests.Session()
        r = requests.packages.urllib3.util.retry.Retry(total=3, backoff_factor=0.5)
        a = requests.adapters.HTTPAdapter(max_retries=r)
        s.mount('http://', a)
        return s
    reply = requests_retry().post(
        'http://juggler-push.search.yandex.net/events',
        json={
            'source': 'sandbox',
            'events': [
                {
                    'host': host,
                    'service': service,
                    'status': status,
                    'description': description,
                    'tags': tags
                }
            ]
        },
        timeout=10,
    )
    logging.info('Juggler event (service: {}; status: {}) returned {} code: {}'.format(service, status, reply.status_code, reply.text))


def retry(ExceptionToCheck=Exception, tries=3, delay=1, backoff=2):
    """Retry calling the decorated function using an exponential backoff.

    http://www.saltycrane.com/blog/2009/11/trying-out-retry-decorator-python/
    original from: http://wiki.python.org/moin/PythonDecoratorLibrary#Retry

    :param ExceptionToCheck: the exception to check. may be a tuple of
        exceptions to check
    :type ExceptionToCheck: Exception or tuple
    :param tries: number of times to try (not retry) before giving up
    :type tries: int
    :param delay: initial delay between retries in seconds
    :type delay: int
    :param backoff: backoff multiplier e.g. value of 2 will double the delay
        each retry
    :type backoff: int
    """
    def deco_retry(f):

        @wraps(f)
        def f_retry(*args, **kwargs):
            mtries, mdelay = tries, delay
            logging.debug(
                'retry params: tries=%d, delay=%d, backoff=%d',
                tries,
                delay,
                backoff,
            )
            while mtries > 1:
                try:
                    return f(*args, **kwargs)
                except ExceptionToCheck as e:
                    logging.exception(e)
                    logging.warn('Retrying in %d seconds...', mdelay)
                    time.sleep(mdelay)
                    mtries -= 1
                    mdelay *= backoff
            return f(*args, **kwargs)

        return f_retry

    return deco_retry
