import logging
import os
import shutil

from sandbox import common
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
from sandbox import sdk2

from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6
from sandbox.projects.statinfra.statinfra_task.tools import mkdir_p


class StatinfraDockerContentResourse(sdk2.Resource):
    releasable = False
    restart_policy = ctr.RestartPolicy.DELETE


class StatinfraBuildDockerImageFromGit(sdk2.Task):
    checkout_path = 'src'

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Git parameters') as git_block:
            repository = sdk2.parameters.String('Repository', required=True)
            git_branch = sdk2.parameters.String('Branch', default='master')
            git_tag = sdk2.parameters.String('Tag', required=True)
            path_to_dockerfile = sdk2.parameters.String('Path to dockerfile inside the repository', default='.')
            ssh_vault_name = sdk2.parameters.String('Vault item with ssh key for git access')
            ssh_vault_owner = sdk2.parameters.String('Vault item owner')

        with sdk2.parameters.Group('Registry parameters') as docker_block:
            registry_tags = sdk2.parameters.List('Tags to publish image with (registry.yandex.net/<this tags>)')
            oauth_vault_name = sdk2.parameters.String(
                'Vault item with oauth token for '
                'registry.yandex.net (vault item name)'
            )
            oauth_vault_owner = sdk2.parameters.String('Vault item owner')
            registry_login = sdk2.parameters.String('Yandex login to use with docker login')
            cache_from_image_tag = sdk2.parameters.String(
                '--cache-from image tag (if present, will be used as docker build --cache-from=<this>)'
            )

    def git_clone(self):
        with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            if os.path.exists(self.checkout_path):
                shutil.rmtree(self.checkout_path)

            mkdir_p(self.checkout_path)

            sp.check_call(
                [
                    'git', 'clone', '-b', self.Parameters.git_branch,
                    self.Parameters.repository, self.checkout_path
                ]
            )
            sp.check_call(['git', 'checkout', self.Parameters.git_tag], cwd=self.checkout_path)
            sp.check_call(['git', 'submodule', 'init'], cwd=self.checkout_path)
            sp.check_call(['git', 'submodule', 'update'], cwd=self.checkout_path)

    def prepare_resource(self):
        self.git_clone()
        path = os.path.join(self.checkout_path, self.Parameters.path_to_dockerfile)
        if not os.path.exists(path):
            raise common.errors.TaskError('Directory {} not found inside the repository'.format(
                self.Parameters.path_to_dockerfile)
            )
        resource = StatinfraDockerContentResourse(
            self,
            description='Dockerfile and contents from {repo}'.format(
                repo=self.Parameters.repository,
            ),
            path=path,
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def build_image(self, resource):
        task_class = sdk2.Task['BUILD_DOCKER_IMAGE_V6']
        kwargs = {
            BuildDockerImageV6.PackagedResource.name: resource.id,
            BuildDockerImageV6.RegistryTags.name: self.Parameters.registry_tags,
            BuildDockerImageV6.RegistryLogin.name: self.Parameters.registry_login,
            BuildDockerImageV6.VaultItemName.name: self.Parameters.oauth_vault_name,
            BuildDockerImageV6.VaultItemOwner.name: self.Parameters.oauth_vault_owner,
            BuildDockerImageV6.CacheFromImageTag.name: self.Parameters.cache_from_image_tag
        }
        logging.info(str(kwargs))
        sub_task = task_class(
            task_class.current,
            description='Building and publishing Docker image from Git source (by {})'.format(self.id),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        self.Context.sub_task_id = sub_task.id
        raise sdk2.WaitTask([sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.first_step:
            resource = self.prepare_resource()
            self.build_image(resource)
        with self.memoize_stage.second_step:
            sub_task = self.find(id=self.Context.sub_task_id).first()
            if sub_task.status not in ctt.Status.Group.SUCCEED:
                raise common.errors.TaskFailure('Subtask is failed with status {}'.format(sub_task.status))
