import six
import os
import json
import datetime
import re
import logging
import time

import requests

from sandbox import sdk2
from sandbox import common
from sandbox.common.utils import server_url

from . import config
from .tools import silently


def send_notifications(parameters, failed, stdouterr, action_info, start_time, task_id):
    send_to_juggler(
        action_id=parameters.action_id,
        event_params=parameters.event_params,
        failed=failed,
        stdouterr=stdouterr,
        action_info=action_info,
        task_id=task_id
    )

    if failed:
        send_failure_to_sentry(
            action_id=parameters.action_id,
            event_params=parameters.event_params,
            stdouterr=stdouterr,
            start_time=start_time,
        )


@silently
def send_to_juggler(action_id, event_params, failed, stdouterr, action_info, task_id):
    if failed and stdouterr:
        cut_stdouterr = six.text_type('{beginnig}\n...\n{end}'.format(
            beginnig=stdouterr[:340],
            end=stdouterr[-340:]
        ), errors='ignore')

    tags = []
    custom_tags = action_info.get('juggler_tags', [])
    try:
        tags += custom_tags
    except Exception:
        logging.exception('Failed to add custom juggler tags')

    reply = requests.post(
        'http://juggler-push.search.yandex.net/events',
        json={
            'source': 'sandbox',
            'events': [
                {
                    'host': 'StatinfraMonitoring',
                    'service': '{env}_{a_id}'.format(
                        env=config.si_env,
                        a_id=action_id.replace(':', '_'),
                    ),
                    'status': 'CRIT' if failed else 'OK',
                    'description': 'task: {server_url}/task/{task_id}, params: {params}, err: {err}'.format(
                        server_url=server_url(),
                        task_id=task_id,
                        params=event_params,
                        err=cut_stdouterr if failed else '',
                    ),
                    'tags': tags,
                }
            ]
        },
        timeout=10,
    )
    event_status = reply.json()["events"][0]
    if event_status["code"] != 200:
        raise Exception(event_status["error"])


@silently
def send_failure_to_sentry(action_id, event_params, stdouterr, start_time):
    import yaml
    masks_path = os.path.dirname(os.path.realpath(__file__)) + "/" + 'masks.yaml'
    masks = yaml.load(open(masks_path))['masks']

    import raven
    dsn = sdk2.Vault.data('STATINFRA', 'statinfra_sentry_actions_dsn')
    sentry_client = raven.Client(dsn=dsn, raise_send_errors=True)

    task_params_masked = json.dumps(event_params or {})
    for mask in masks:
        task_params_masked = re.sub(mask['pattern'], mask['replace'], task_params_masked)

    def extract_external_failure_message(msg):
        """ External actions error messages always starts with 2 lines
            of useless (in our case) staff.
            Except of cases like forced termination of task.
        """
        failure = msg.splitlines(True)
        if len(failure) > 2 and \
                not failure[2].startswith('Traceback (most recent call last)'):
            return ''.join(failure[2:])
        else:
            return msg

    unixtime = int(time.time())
    message = extract_external_failure_message(stdouterr)
    if message:
        sentry_client.send(
            message=extract_external_failure_message(stdouterr),
            culprit=action_id,
            timestamp=datetime.datetime.utcfromtimestamp(unixtime),
            tags={'action_id': action_id},
            fingerprint=[action_id, task_params_masked],
            extra={
                'task_params': event_params,
                'task_params_masked': task_params_masked,
                'server_name': common.config.Registry().this.id,
                'start_time': start_time,
                'fail_time': datetime.datetime.fromtimestamp(unixtime).strftime("%Y-%m-%d %H:%M:%S")
            }
        )
