import re
import logging
import hashlib
import json
import datetime

from sandbox.sdk2.helpers import subprocess as sp

from . import config

import sandbox.projects.statinfra
STATINFRA_ACTION_REPOS = getattr(sandbox.projects.statinfra, config.ACTION_REPOS_RESOURCE_NAME)
STATINFRA_JOB_REPOS = getattr(sandbox.projects.statinfra, config.JOB_REPOS_RESOURCE_NAME)


LEGACY_TYPE = 'legacy'


def get_action_hash(action_id, params):
    return hashlib.md5('{}-{}'.format(action_id, json.dumps(params, sort_keys=True))).hexdigest()


def restart_action(self, stdouterr, failed):
    if failed:
        next_start_time = get_restart_time_on_failure(self.Context.action_info)
    else:
        next_start_time = get_restart_time_on_success(self.Context.action_info, stdouterr)

    clone_into_new_task(self, next_start_time, failed)


def get_restart_time_from_output(sp_out):
    m_perl = re.search(r'(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})$', sp_out)
    m_python = re.search(
        r"(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}) (I know what I'm doing)\n",
        sp_out
    )
    m = m_perl or m_python
    if m:
        next_start_time = m.group(1)
        return next_start_time


def get_restart_time_on_success(action_info, stdouterr):
    if action_info['action_type'] == LEGACY_TYPE:
        return get_restart_time_from_output(stdouterr)
    interval = action_info['run_interval']
    next_start_time = (
        datetime.datetime.now() +
        datetime.timedelta(seconds=interval)
    ).strftime('%Y-%m-%d %H:%M:%S')
    return next_start_time


def get_restart_time_on_failure(action_info):
    default_interval = 10 * 60
    if action_info['action_type'] != LEGACY_TYPE:
        interval = (
            action_info['run_interval_on_failure'] or
            action_info['run_interval'] or
            default_interval
        )
        restart_time = (datetime.datetime.now() + datetime.timedelta(seconds=interval)).strftime('%Y-%m-%d %H:%M:%S')
        return restart_time
    try:
        out = sp.check_output(
            ['/usr/bin/statbox-ctl.pl info-action-start-time-on-failure "%s"' % action_info['action_id']],
            shell=True, stderr=sp.STDOUT
        )
    except sp.CalledProcessError:
        logging.warning('Returning default next start time')
        return (datetime.datetime.now() + datetime.timedelta(minutes=10)).strftime('%Y-%m-%d %H:%M:%S')
    return get_restart_time_from_output(out)


def clone_into_new_task(self, start_time, failed):
    if self.Parameters.run_limit != -1:
        if (
                int(self.Parameters.statbox_exec_stat['failures']) +
                int(self.Parameters.statbox_exec_stat['successes']) +
                1
        ) >= self.Parameters.run_limit:
            self.set_info('run_limit is reached, no restart')
            return

    if failed:
        cont_failures = int(self.Parameters.statbox_exec_stat['cont_failures'])
        if self.Parameters.max_cont_failures != -1 and cont_failures + 1 >= self.Parameters.max_cont_failures:
            self.set_info('cont_failures limit is reached, no restart')
            return

    unique_key = get_action_hash(self.Parameters.action_id, self.Parameters.event_params)

    new_statbox_exec_stat = dict(self.Parameters.statbox_exec_stat)
    if failed:
        new_statbox_exec_stat['failures'] = int(new_statbox_exec_stat['failures']) + 1
        new_statbox_exec_stat['cont_failures'] = int(new_statbox_exec_stat['cont_failures']) + 1
        new_statbox_exec_stat['cont_successes'] = 0
    else:
        new_statbox_exec_stat['successes'] = int(new_statbox_exec_stat['successes']) + 1
        new_statbox_exec_stat['cont_successes'] = int(new_statbox_exec_stat['cont_successes']) + 1
        new_statbox_exec_stat['cont_failures'] = 0

    request_data = {
        'source': self.id,
        'owner': self.owner,
        'priority': {
            'class': 'SERVICE',
            'subclass': 'HIGH',
        },
        'uniqueness': {
            'key': unique_key,
            'excluded_statuses': ['BREAK', 'FINISH', 'EXECUTE'],
        },
        'custom_fields': [
            {
                'name': 'action_id',
                'value': self.Parameters.action_id
            },
            {
                'name': 'event_params',
                'value': self.Parameters.event_params
            },
            {
                'name': 'statbox_exec_stat',
                'value': new_statbox_exec_stat
            },
            {
                'name': 'start_time',
                'value': start_time
            },
            {
                'name': 'primary_sandbox_task_id',
                'value': getattr(self.Parameters.primary_sandbox_task_id, 'id', self.Parameters.primary_sandbox_task_id)
            },
            {
                'name': 'container',
                'value': None
            },
            {
                'name': 'action_repos',
                'value': None
            },
            {
                'name': 'job_repos',
                'value': None
            }

        ]
    }

    if self.Requirements.semaphores is not None:
        request_data['requirements'] = {'semaphores': self.Requirements.semaphores.to_dict()}

        # try to update semaphores
        if not failed:
            try:
                task_info = self.Context.action_info
                semaphores = task_info['semaphores']
                if semaphores:
                    request_data['requirements']['semaphores'] = {
                        'acquires': [
                            {'name': sem_name, 'capacity': 1}
                            for sem_name in semaphores
                        ]
                    }
            except Exception:
                logging.exception('Failed to fetch semaphores from statinfra api')
                self.set_info('WARNING: Failed to fetch semaphores from statinfra api')

    new_task = self.server.task(request_data)
    new_task_id = new_task['id']
    if new_task['status'] != 'DRAFT':
        self.set_info('DEDUPLICATION: task %s already exists' % new_task_id)
        return

    self.set_info('Planning myself on {nst}, task id: {tid}, uniqueue_key: {uk}'.format(
        nst=start_time, tid=new_task_id, uk=unique_key
    ))

    with self.memoize_stage.cloning:
        self.Context.new_task_id = new_task_id
        response = self.server.batch.tasks.start.update([new_task_id])
        self.set_info('Response: %s' % response)


def is_cyclic_action(action_info, stdouterr):
    if action_info['action_type'] == LEGACY_TYPE:
        next_start_time = get_restart_time_from_output(stdouterr)
        if next_start_time:
            return True
        return False
    if action_info['run_interval'] is not None:
        return True
    return False


def action_deleted_from_repo(stdouterr, action_id):
    marks = (
        'actionId \'{}\' isn\'t known'.format(action_id),
        'No such file or directory at / usr / share / perl5 / Statbox / XActionIdempotent.pm',
        'There is no such {}'.format(action_id),
    )

    deleted = any(m in stdouterr for m in marks)

    if deleted:
        logging.warning('Action %s has been deleted from repo', action_id)

    return deleted
