import json
import logging
import os
import re
import requests
import urlparse
import yaml

from requests.packages.urllib3.util import Retry
from requests.adapters import HTTPAdapter
from simplejson.decoder import JSONDecodeError
from copy import deepcopy

from .config import STATINFRA_API_HOST

_logger = logging.getLogger(__name__)

NILE_REGEXP = re.compile(r'Nile script failed\. Exit code ([-]?\d+)\. Fail class: (.+)\n')
YQL_REGEXP = re.compile(r'YQL query failed\. Task fate: (\d+). Fail class: (.+)\n')
QC_REGEXP = re.compile(r'action `external:stjob-regular/qcregular` child exited with code (\d+)')
CUSTOM_REGEXP = re.compile(r'Custom script failed. Exit code (\d+)')
SENTRY_SEND_BY = 'task-inspector'
LOG_ERROR_REGEXP = re.compile('contains error:(.*)$', flags=re.MULTILINE)
LOG_CLEAN_REGEXPS = [
    re.compile(
        r'^.*JobRunningError: Subprocess exit with code \d+ and stderror: ',
        flags=re.MULTILINE,
    ),
    re.compile(
        r'^My PID=\d+, command=mrproc-job-run .+?\n',
        flags=re.MULTILINE,
    ),
    re.compile(r'^\s*\(in\s+cleanup\)\s*\[.*?\]\s*', flags=re.MULTILINE),
    re.compile(r'^\s*\[.*?\]\s*', flags=re.MULTILINE),
    re.compile(r'^\s*\(in\s+cleanup\)\s*', flags=re.MULTILINE),
    re.compile(r'^\s*', flags=re.MULTILINE),
    re.compile(
        r'(\\n|\n)'.join([
            r'Sentry is attempting to send \d+ pending error messages',
            r'Waiting up to \d+ seconds',
            r'Press Ctrl-C to quit',
            r'',
        ]),
        flags=re.MULTILINE,
    ),
]
API_URL = 'api/v1/demand_registry'
STOP_TASK = 'stop'
RESTART_TASK = 'restart'


class TaskFate(object):

    def __init__(self, action, fail_class, details, content, rule):
        self.action = action
        self.fail_class = fail_class
        self.details = details
        self.content = content
        self.rule = rule

    def serialize(self, cluster):
        data = {
            'action': self.action,
            'class': self.fail_class,
            'details': self.details,
            'content': self.content[:3300],
            'rule': self.rule,
            'cluster': cluster,
        }
        return data


class TaskInspector(object):

    def __init__(self, task, stdouterr):
        self.task = task
        self.error = stdouterr

    def run(self):
        rules = self.get_rules()
        task_fate = self.get_task_fate(rules)
        return self.manage_task(task_fate)

    def manage_task(self, task_fate):
        def stop_task():
            _logger.info('Task was stopped by TaskInspector')
            self.demand_fail(description=task_fate.serialize(self.task.Parameters.event_params.get('cluster')))
            return STOP_TASK

        def continue_task():
            _logger.info("Task wasn't stopped by TaskInspector")
            return RESTART_TASK

        if task_fate.action == 'continue':
            return continue_task()
        if task_fate.action == 'retry-then-stop':
            if int(self.task.Parameters.statbox_exec_stat['cont_failures']) < 3:
                return continue_task()
            else:
                return stop_task()
        if task_fate.action != 'stop':
            _logger.warning('Unknown action %s. Stop task.', task_fate.action)
        return stop_task()

    def get_task_fate(self, rules):
        if self.task.Parameters.max_cont_failures != -1:
            return TaskFate(
                action='continue',
                fail_class='unknown',
                details='unknown_error',
                content=self.error,
                rule=0,
            )
        self.clean_error()
        nile_match = NILE_REGEXP.search(self.error)
        if nile_match:
            exit_code, fail_class = nile_match.group(1), nile_match.group(2)
            return TaskFate(
                action=self.get_action_by_exit_code(exit_code),
                fail_class=fail_class,
                details='Nile job fail by {0} error'.format(fail_class),
                content=self.error,
                rule=44,
            )
        yql_match = YQL_REGEXP.search(self.error)
        if yql_match:
            exit_code, fail_class = yql_match.group(1), yql_match.group(2)
            return TaskFate(
                action=self.get_action_by_exit_code(exit_code),
                fail_class=fail_class,
                details='YQL query fail by {0} error'.format(fail_class),
                content=self.error,
                rule=52,
            )
        qc_match = QC_REGEXP.search(self.error)
        if qc_match:
            exit_code = qc_match.group(1)
            return TaskFate(
                action=self.get_action_by_exit_code(exit_code),
                fail_class='qc',
                details='QC job fail',
                content=self.error,
                rule=50,
            )
        custom_match = CUSTOM_REGEXP.search(self.error)
        if custom_match:
            exit_code = custom_match.group(1)
            return TaskFate(
                action=self.get_action_by_exit_code(exit_code),
                fail_class='custom',
                details='Custom job fail',
                content=self.error,
                rule=51,
            )
        for rule in rules:
            for regexp, value in rule.iteritems():
                if re.search(regexp, self.error, flags=re.MULTILINE):
                    if isinstance(value, list):
                        return self.get_task_fate(value)
                    return TaskFate(
                        action=value['action'],
                        fail_class=value['class'],
                        details=value['details'],
                        content=self.error,
                        rule=value['rule'],
                    )
        _logger.warning('Error does not much any rules: {0}'.format(self.error))
        return TaskFate(
            action='continue',
            fail_class='unknown',
            details='unknown_error',
            content=self.error,
            rule=0,
        )

    def get_rules(self):
        dirname, _ = os.path.split(os.path.abspath(__file__))
        with open(os.path.join(dirname, 'rules.yaml')) as f:
            rules = yaml.load(f)
        return rules

    def clean_error(self):
        match = LOG_ERROR_REGEXP.search(self.error)
        if match:
            self.error = match.group(1)
        for regexp in LOG_CLEAN_REGEXPS:
            self.error = regexp.sub('', self.error)

    def get_action_by_exit_code(self, code):
        try:
            code = int(code)
        except ValueError:
            return 'stop'
        return {
            1: 'stop',
            2: 'continue',
            3: 'retry-then-stop',
            -9: 'stop',
        }.get(code, 'stop')

    def demand_fail(self, description):
        status_description = str(description)
        try:
            status_description = json.dumps(description)
        except Exception:
            _logger.warning('Can\'t decode status description: %s', description)
        self.change_demand_status(
            status='failed',
            data={
                'status_description': status_description,
            }
        )
        _logger.info("Fail demand %s on cluster %s with description %s",
                     self.task.Parameters.event_params.get('demand'),
                     self.task.Parameters.event_params.get('cluster'),
                     description,
                     )

    def change_demand_status(self, status, data):
        relative_url = '/'.join([API_URL, self.task.Parameters.event_params['demand'], status])
        response = self.request('put', relative_url, json=data)
        _logger.info(
            'Change status request return response: %s',
            response.content,
        )
        if response.status_code == 400:
            error = response.content
            try:
                error = response.json()
            except (ValueError, JSONDecodeError):
                _logger.warning(
                    'Can\'t parse json, use content: %s',
                    error,
                )
                response.raise_for_status()
            if error.get('error', '') == 'Invalid status':
                _logger.warning(
                    'Can\'t set new status %s: %s',
                    status,
                    error,
                )
                return
        response.raise_for_status()

    def request(self, method, relative_url, **kwargs):
        url = urlparse.urljoin(STATINFRA_API_HOST, relative_url)
        session = requests.Session()
        session = self.add_requests_retry(
            session=session,
            total=5,
            backoff_factor=2,
            status_forcelist=[500, 502, 503, 504, 521],
            redirect=10,
        )
        return session.request(
            method,
            url,
            stream=False,
            timeout=30,
            **kwargs
        )

    def add_requests_retry(self, session, **kwargs):
        session = deepcopy(session)
        for prefix in ['http://', 'https://']:
            session.mount(
                prefix,
                HTTPAdapter(
                    max_retries=Retry(**kwargs),
                ),
            )
        return session
