# coding: utf8
import six
import json
import logging
import shutil
import tarfile
import tempfile

import datetime
import os
import pytz
import re
import requests
import sandbox.agentr.types
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.projects.statinfra
import time
import types
from requests.exceptions import ReadTimeout
from sandbox.agentr.errors import InvalidImage
from sandbox.common.errors import (
    TaskFailure,
    TemporaryError,
    VaultNotFound,
    VaultNotAllowed,
    InvalidResource,
)
from sandbox.common.types.task import TaskTag, Semaphores
from sandbox.projects.common import solomon
from sandbox.projects.common import binary_task
from sandbox.sdk2.helpers import subprocess as sp

from sandbox import common
from sandbox import sandboxsdk
from sandbox import sdk2
from . import config
from . import tools
from .notifications import send_notifications
from .restart import restart_action, is_cyclic_action, action_deleted_from_repo
from .task_execution_log import write_task_execution_log, hide_oauth
from .task_inspector import TaskInspector, RESTART_TASK
from ..extra import chdir, retry

STATINFRA_LXC_CONTAINER = getattr(sandbox.projects.statinfra, config.LXC_CONTAINER_RESOURCE_NAME)
STATINFRA_ACTION_REPOS = getattr(sandbox.projects.statinfra, config.ACTION_REPOS_RESOURCE_NAME)
STATINFRA_JOB_REPOS = getattr(sandbox.projects.statinfra, config.JOB_REPOS_RESOURCE_NAME)
STATINFRA_LIB_REPOS = getattr(sandbox.projects.statinfra, config.LIB_REPOS_RESOURCE_NAME)
STATINFRA_CONFIG_REPOS = getattr(sandbox.projects.statinfra, config.CONFIG_REPOS_RESOURCE_NAME)
STATINFRA_VENVS = getattr(sandbox.projects.statinfra, config.VENVS_RESOURCE_NAME)

JOB_RUNNER_ID = 'external:statinfra-action/rjm/job-runner'
COPY_TABLE_ID = 'external:statinfra-action/rjm/copy-table'
SECRETS_ENV_PREFIX = 'STATINFRA_SECRET'
DEFAULT_SEMAPHORE = 'statinfra-task'
ACTION_STDOUTERR_FILENAME = 'action_stdouterr.txt'

stats = dict()


def track(f):
    def tracking(*args, **kwargs):
        global stats
        a = time.time()
        _return = f(*args, **kwargs)
        b = time.time()
        f_id = f.__name__
        stats[f_id] = b - a
        msg = 'TRACKING function {} took {:f} second(s)'.format(f_id, stats[f_id])
        logging.info(msg)
        return _return
    return tracking


def solomon_push():
    """Send sensor values to solomon"""
    global stats
    token = sdk2.Vault.data('STATINFRA', 'ROBOT_STATINFRA_SOLOMON_TOKEN')
    logging.info("stats is " + str(stats))
    sensors = solomon.create_sensors(stats)
    params = dict(
        project='statbox',
        cluster='sandbox',
        service=config.task_type
    )
    try:
        solomon.push_to_solomon_v2(token, params, sensors, common_labels=dict())
    except Exception as e:
        logging.error('Push to solomon failed:\n{}'.format(e))


class StatinfraTaskFailure(TaskFailure):
    pass


class StatinfraTask(sdk2.Task, binary_task.LastBinaryTaskRelease):
    name = config.task_type
    PRIO_COEFFICIENT = 10 ** 15

    class Requirements(sdk2.Task.Requirements):
        disk_space = 8000
        cores = 1
        environments = (
            sandboxsdk.environments.PipEnvironment('pytz'),
            sandboxsdk.environments.PipEnvironment('raven'),
        )
        dns = ctm.DnsType.DNS64
        ram = 8 * 1024
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 2048, None)

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        _lbrp = binary_task.binary_release_parameters(none=True)

        max_restarts = 15
        kill_timeout = 60 * 60 * 6
        custom_kill_timeout = sdk2.parameters.Integer('Custom kill timeout', default=0)

        action_id = sdk2.parameters.String('Action ID', required=True)

        event_params = sdk2.parameters.Dict('Event params')

        extra_resource_ids = sdk2.parameters.Resource('Extra resource ids', default=[], multiple=True)
        extra_resource_names = sdk2.parameters.List('Extra resource names', default=[])

        start_time = sdk2.parameters.String('Start time (%Y-%m-%d %H:%M:%S)')

        container = sdk2.parameters.Container(
            'LXC Container',
            resource_type=STATINFRA_LXC_CONTAINER,
            required=True,
            register_dependency=False,
        )

        action_repos = sdk2.parameters.Resource(
            'Statinfra action repos',
            resource_type=STATINFRA_ACTION_REPOS,
            required=True,
            register_dependency=False,
        )

        job_repos = sdk2.parameters.Resource(
            'Statinfra job repos',
            resource_type=STATINFRA_JOB_REPOS,
            required=True,
            register_dependency=False,
        )

        lib_repos = sdk2.parameters.Resource(
            'Statinfra lib repos',
            resource_type=STATINFRA_LIB_REPOS,
            # required=True,
            register_dependency=False,
        )

        config_repos = sdk2.parameters.Resource(
            'Statinfra config repos',
            resource_type=STATINFRA_CONFIG_REPOS,
            required=True,
            register_dependency=False,
        )

        venv_repos = sdk2.parameters.Resource('Statinfra venvs', resource_type=STATINFRA_VENVS)

        privileged = sdk2.parameters.Bool('Run under root privileges', default=False)

        primary_sandbox_task_id = sdk2.parameters.Task('ID of first task', default=None)

        statbox_exec_stat = sdk2.parameters.Dict(
            'Statbox execution info',
            default={
                'failures': 0,
                'cont_failures': 0,
                'successes': 0,
                'cont_successes': 0
            }
        )

        prio = sdk2.parameters.Float('Priority', default=110.0)

        options = sdk2.parameters.Dict('Options')

        step_events = sdk2.parameters.String('STEP events', description='STEP events in json')
        event_deps_case = sdk2.parameters.String('STEP event_dependencies case', default='default')

        max_cont_failures = sdk2.parameters.Integer('Max cont_failures', default=-1)
        failure_interval_factor = sdk2.parameters.Integer('Failure interval factor', default=1)
        run_limit = sdk2.parameters.Integer('Run limit', default=-1)

    def is_job_runner(self):
        return self.Parameters.action_id == JOB_RUNNER_ID

    def is_copy_table(self):
        return self.Parameters.action_id == COPY_TABLE_ID

    def on_create(self):
        self.Parameters.venv_repos = STATINFRA_VENVS.find(state=ctr.State.READY).first().id

        self.Parameters.lib_repos = STATINFRA_LIB_REPOS.find(state=ctr.State.READY).first().id

        self.Parameters.action_repos = STATINFRA_ACTION_REPOS.find(state=ctr.State.READY).first().id

        self.Parameters.job_repos = STATINFRA_JOB_REPOS.find(state=ctr.State.READY).first().id

        self.Parameters.config_repos = STATINFRA_CONFIG_REPOS.find(state=ctr.State.READY).first().id

        self.Parameters.container = STATINFRA_LXC_CONTAINER.find(
            state=ctr.State.READY,
            owner=config.LXC_CONTAINER_RESOURCE_OWNER
        ).first().id

    def _test_valid_tag(self, tag):
        try:
            return TaskTag.test(tag)
        except ValueError:
            return False

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)

        if self.Parameters.privileged:
            self.Requirements.privileged = True

        if self.Parameters.primary_sandbox_task_id is None:
            self.Parameters.primary_sandbox_task_id = self.id

        if not self.Parameters.venv_repos:
            self.Parameters.venv_repos = STATINFRA_VENVS.find(state=ctr.State.READY).first().id

        if not self.Parameters.lib_repos:
            self.Parameters.lib_repos = STATINFRA_LIB_REPOS.find(state=ctr.State.READY).first().id

        if not self.Parameters.action_repos:
            self.Parameters.action_repos = STATINFRA_ACTION_REPOS.find(state=ctr.State.READY).first().id

        if not self.Parameters.job_repos:
            self.Parameters.job_repos = STATINFRA_JOB_REPOS.find(state=ctr.State.READY).first().id

        if not self.Parameters.config_repos:
            self.Parameters.config_repos = STATINFRA_CONFIG_REPOS.find(state=ctr.State.READY).first().id

        if not self.Parameters.container:
            self.Parameters.container = STATINFRA_LXC_CONTAINER.find(
                state=ctr.State.READY,
                owner=config.LXC_CONTAINER_RESOURCE_OWNER
            ).first().id

        try:
            r = requests.get(
                '{}/api/v1/action_registry/{}'.format(config.STATINFRA_API_HOST, self.Parameters.action_id),
                timeout=10,
            )
            r.raise_for_status()
        except Exception:
            logging.exception('Failed to get action info in on_save')
            self.Context.action_info = {}
        else:
            action_info = r.json()
            if action_info['status'] != 'active':
                logging.exception('Action has bad status %s', action_info['status'])
                self.Context.action_info = {}
            else:
                self.Context.action_info = action_info

        if self.Parameters.action_id:
            sandbox_tags = self.Context.action_info.get('sandbox_tags', [])
            tags = [self.Parameters.action_id] + sandbox_tags
            for word in ('cluster', 'mrbasename', 'mapreduce_base_name'):
                if word in self.Parameters.event_params:
                    cluster = self.Parameters.event_params[word].split('.')[0].replace('yt_', '')
                    tags.append(cluster)

            for word in ('date', 'job', 'scale', 'logname', 'log_type', 'group', 'job_type'):
                if word in self.Parameters.event_params:
                    tags.extend(self.Parameters.event_params[word].split(','))

            if 'timestamp' in self.Parameters.event_params:
                tstamp = self.Parameters.event_params['timestamp']
                m = re.match(r'^(\d{4}-\d{2}-\d{2})', tstamp)
                if m:
                    tags.append(m.group(1))
            valid_tags = [
                tag
                for tag in tags
                if tag and self._test_valid_tag(tag)
            ]
            invalid_tags = list(set(tags) - set(valid_tags))
            if invalid_tags:
                logging.warning('We have found invalid tags: %s. Skip it', invalid_tags)
            self.Parameters.tags = valid_tags

        if len(self.Parameters.extra_resource_ids) != len(self.Parameters.extra_resource_names):
            raise Exception('extra_resource_ids and extra_resource_names have different length')

        self.Parameters.max_cont_failures = self.Context.action_info.get('max_cont_failures', -1)
        if self.is_job_runner():
            self._update_job_info(job_id=self.Parameters.event_params['job'])
            self.Parameters.max_cont_failures = self.Context.job_info['meta'].get('max_cont_failures', -1)
        self.Parameters.failure_interval_factor = self.Context.action_info.get('failure_interval_factor', 1)
        if self.Parameters.custom_kill_timeout:
            self.Parameters.kill_timeout = self.Parameters.custom_kill_timeout
        else:
            self.Parameters.kill_timeout = self.Context.action_info.get('exec_timeout') or 60 * 60 * 6

    def on_timeout(self, prev_status):
        self._copy_python_statinfra_log()
        restart_action(self, stdouterr='', failed=True)
        send_notifications(
            parameters=self.Parameters,
            failed=True,
            stdouterr='Task has been restarted due to timeout',
            action_info=self.Context.action_info,
            start_time=self.Context.start_time,
            task_id=self.id
        )

    @track
    @retry(ReadTimeout)
    def _update_action_info(self):
        r = requests.get(
            '{}/api/v1/action_registry/{}'.format(config.STATINFRA_API_HOST, self.Parameters.action_id),
            timeout=10,
        )
        if r.status_code == 404:
            raise TaskFailure('Seems like action hasn\'t existed in registry')
        try:
            r.raise_for_status()
        except Exception:
            logging.exception('Failed to get action info in on_execute')
            raise TemporaryError('Failed to get action info in on_execute, see logs')
        else:
            action_info = r.json()
            if action_info['status'] != 'active':
                raise TemporaryError('Action has bad status %s', action_info['status'])
            self.Context.action_info = action_info

    @track
    def _update_job_info(self, job_id):
        if job_id is None:
            logging.info('job_id has not been set')
            self.Context.job_info = {}
            return
        url = '{}/api/v1/job_registry/jobs/{}'.format(config.STATINFRA_API_HOST, job_id)
        logging.info('Url for job registry: %s', url)
        r = requests.get(
            url,
            timeout=10,
        )
        if r.status_code == 404:
            raise TaskFailure('Seems like job hasn\'t existed in registry')
        try:
            r.raise_for_status()
        except Exception:
            logging.exception('Failed to get job info in on_execute')
            raise TemporaryError('Failed to get action info in on_execute, see logs')
        else:
            job_info = r.json()
            if job_info['status'] != 'active':
                raise TaskFailure('Job has bad status %s', job_info['status'])
            self.Context.job_info = job_info

    def _safe_format_semaphore(self, sem):
        try:
            sem_dict = sem.to_dict()
            return Semaphores.Acquire(
                name=sem_dict['name'].format(**self.Parameters.event_params),
                capacity=sem_dict['capacity'],
                weight=sem_dict['weight']
            )
        except KeyError:
            return sem

    def on_enqueue(self):
        self.Parameters.score = int(tools.stat_prio_to_pw(self.Parameters.prio) * self.PRIO_COEFFICIENT)
        if self.Parameters.owner != config.OWNER:
            raise TaskFailure('Only STATINFRA group can run STATINFRA_TASK')
        default_semaphore = Semaphores.Acquire(name=DEFAULT_SEMAPHORE, weight=1)
        if self.Requirements.semaphores is None:
            self.Requirements.semaphores = Semaphores(
                acquires=[default_semaphore]
            )
        else:
            if default_semaphore not in self.Requirements.semaphores.acquires:
                self.Requirements.semaphores = Semaphores(
                    acquires=self.Requirements.semaphores.acquires + (default_semaphore,),
                    release=self.Requirements.semaphores.release
                )
            self.Requirements.semaphores = Semaphores(
                acquires=tuple(set(map(self._safe_format_semaphore, self.Requirements.semaphores.acquires))),
                release=self.Requirements.semaphores.release
            )
        MAX_WAIT_SEC = 24 * 60 * 60
        with self.memoize_stage.wait_on_enqueue:
            start_time = self.Parameters.start_time
            if start_time:
                tz = pytz.timezone('Europe/Moscow')
                start_time_obj = datetime.datetime.strptime(start_time, '%Y-%m-%d %H:%M:%S')
                now = datetime.datetime.now(tz).replace(tzinfo=None)
                if start_time_obj > now:
                    wait_sec = (start_time_obj - now).total_seconds()
                    cont_failures = int(self.Parameters.statbox_exec_stat['cont_failures'])
                    if cont_failures >= 1:
                        wait_sec *= self.Parameters.failure_interval_factor ** (cont_failures - 1)
                        wait_sec = min(wait_sec, MAX_WAIT_SEC)

                    start_time_obj = now + datetime.timedelta(seconds=wait_sec)

                    self.set_info(
                        'Waiting for {st} ({ttw} sec). Now {now}'.format(
                            st=start_time_obj.strftime('%Y-%m-%d %H:%M:%S'),
                            ttw=wait_sec,
                            now=now.strftime('%Y-%m-%d %H:%M:%S')
                        )
                    )
                    raise sdk2.WaitTime(wait_sec)

    @track
    def _prepare_venvs(self):
        venvs = sdk2.ResourceData(self.Parameters.venv_repos)
        logging.info('Mounting venvs')
        try:
            venvs_mount = str(venvs.mount())
        except InvalidImage:
            raise TemporaryError('Failed to mount virtualenv image')
        home_venv = '{}/venv'.format(os.environ['HOME'])
        if os.path.islink(home_venv):
            # Fix for local Sandbox: delete symlink if it already exists.
            # Does not happen on real Sandbox (due to execution in ramdisk).
            # Without it the task goes to EXCEPTION trying to create already
            # existing symlink in os.symlink() below.
            logging.info(
                'Venv symlink %s leading to %s detected, deleting it' %
                (home_venv, os.readlink(home_venv))
            )
            os.remove(home_venv)
        logging.info('Venvs {} symlinking to {}'.format(venvs_mount, home_venv))
        os.symlink(venvs_mount, home_venv)

    @track
    def _prepare_libs(self):
        libs_resource = sdk2.ResourceData(self.Parameters.lib_repos)
        libs_tar = tarfile.open(str(libs_resource.path))
        self.Context.pythonpath = str(tempfile.mkdtemp())
        os.chmod(self.Context.pythonpath, 0o755)
        with chdir(self.Context.pythonpath):
            libs_tar.extractall()

    @track
    @retry(InvalidResource)
    def _prepare_repos(self):
        ramdrive_path = str(self.ramdrive.path) if not isinstance(self.ramdrive, types.NoneType) else tempfile.mkdtemp()
        self.Context.statbox_class_filestorage_path = os.path.join(ramdrive_path, 'statbox-class-filestorage')
        tools.mkdir_p(self.Context.statbox_class_filestorage_path)
        blacklist = ('ActionDatabase', 'Action-MapReduceJob')
        for node in os.listdir('/usr/share/statbox-class-filestorage'):
            if node in blacklist:
                continue
            os.symlink(
                os.path.join('/usr/share/statbox-class-filestorage', node),
                os.path.join(self.Context.statbox_class_filestorage_path, node)
            )
        self._prepare_actions_repos()
        self._prepare_jobs_repos()
        self._prepare_configs_repos()

    @track
    def _prepare_extra_resources(self):
        named_resources = zip(
            self.Parameters.extra_resource_names,
            self.Parameters.extra_resource_ids,
        )
        resource_paths = {}
        with sdk2.helpers.ProcessLog(self, logger='prepare_extra_resources') as pl:
            for name, resource in named_resources:
                resource_data = sdk2.ResourceData(resource)
                resource_paths[name] = str(resource_data.path)
                pl.logger.info('Resource with name %s has path %s', name, resource_data.path)
        return resource_paths

    @track
    def _prepare_actions_repos(self):
        deb_actions_dir = '/usr/share/statbox-class-filestorage/ActionDatabase/external'

        extract_dir = os.path.join(self.Context.statbox_class_filestorage_path, 'ActionDatabase')
        tools.mkdir_p(os.path.join(extract_dir, 'external'))

        actions_resource = self.Parameters.action_repos
        repos_resource_data = sdk2.ResourceData(actions_resource)
        repos_resource_path = repos_resource_data.path

        with sdk2.helpers.ProcessLog(self, logger='tar_actions') as pl:
            return_code = sp.Popen(
                ['tar', '-xvf', str(repos_resource_path), '-C', extract_dir],
                shell=False, stdout=pl.stdout, stderr=sp.STDOUT
            ).wait()
            if return_code != 0:
                raise Exception('Failed to unpack actions repos')

        self._magic_for_move_statkey_actions(deb_actions_dir)

        for repo_name in os.listdir(deb_actions_dir):
            if not os.path.exists(os.path.join(extract_dir, 'external', repo_name)):
                os.symlink(os.path.join(deb_actions_dir, repo_name), os.path.join(extract_dir, 'external', repo_name))

        self._move_statkey_actions(extract_dir)

        for node in os.listdir('/usr/share/statbox-class-filestorage/ActionDatabase'):
            if node == 'external':
                continue
            elif node == 'a':
                for node2 in os.listdir('/usr/share/statbox-class-filestorage/ActionDatabase/a'):
                    os.symlink(
                        os.path.join('/usr/share/statbox-class-filestorage/ActionDatabase/a', node2),
                        os.path.join(extract_dir, 'a', node2)
                    )
                continue
            os.symlink(
                os.path.join('/usr/share/statbox-class-filestorage/ActionDatabase', node),
                os.path.join(extract_dir, node)
            )

    def _magic_for_move_statkey_actions(self, deb_actions_dir):
        '''
        [STATINFRA-10379] remove broken outer-action link created by statbox-mrworker
        '''
        broken_link = os.path.join(deb_actions_dir, 'outer-action')
        if os.path.islink(broken_link):
            try:
                os.remove(broken_link)
                logging.info('Deleted {}'.format(broken_link))
            except OSError as e:
                logging.info('Deleting {}: {}'.format(broken_link, e))
            if not os.path.exists(broken_link):
                try:
                    os.makedirs(broken_link)
                    logging.info('Created {}'.format(broken_link))
                except OSError as e:
                    logging.info('Creating {}: {}'.format(broken_link, e))
        else:
            logging.info('{} is not symlink'.format(broken_link))

    @track
    def _move_statkey_actions(self, extract_dir):
        statkey_actions_src_path = os.path.join(extract_dir, 'a/statkey-action')
        statkey_actions_dst_path = os.path.join(extract_dir, 'external/outer-action/statkey')
        if not os.path.exists(statkey_actions_src_path):
            logging.info('Statkey dir from resource not found. Skip.')
            return
        if os.path.exists(statkey_actions_dst_path):
            logging.info('Remove old dir from deb %s', statkey_actions_dst_path)
            shutil.rmtree(statkey_actions_dst_path)
        tools.mkdir_p(statkey_actions_dst_path)
        for filename in os.listdir(statkey_actions_src_path):
            path = os.path.join(statkey_actions_src_path, filename)
            logging.info('Copy %s from new to old %s', path, statkey_actions_dst_path)
            shutil.move(path, statkey_actions_dst_path)

    @track
    def _prepare_jobs_repos(self):
        deb_jobs_dir = '/usr/share/statbox-class-filestorage/Action-MapReduceJob/jobs/x'

        extract_dir = os.path.join(self.Context.statbox_class_filestorage_path, 'Action-MapReduceJob', 'jobs')
        tools.mkdir_p(os.path.join(extract_dir, 'x'))

        repos_resource = self.Parameters.job_repos
        repos_resource_data = sdk2.ResourceData(repos_resource)
        repos_resource_path = repos_resource_data.path

        with sdk2.helpers.ProcessLog(self, logger='tar_jobs') as pl:
            return_code = sp.Popen(
                ['tar', '-xvf', str(repos_resource_path), '-C', extract_dir],
                shell=False, stdout=pl.stdout, stderr=sp.STDOUT
            ).wait()
            if return_code != 0:
                raise TemporaryError('Failed to unpack jobs repos')

        for repo_name in os.listdir(deb_jobs_dir):
            if not os.path.exists(os.path.join(extract_dir, 'x', repo_name)):
                os.symlink(os.path.join(deb_jobs_dir, repo_name), os.path.join(extract_dir, 'x', repo_name))

        for node in os.listdir('/usr/share/statbox-class-filestorage/Action-MapReduceJob'):
            if node == 'jobs':
                continue
            os.symlink(
                os.path.join('/usr/share/statbox-class-filestorage/Action-MapReduceJob', node),
                os.path.join(self.Context.statbox_class_filestorage_path, 'Action-MapReduceJob', node)
            )

        for node in os.listdir('/usr/share/statbox-class-filestorage/Action-MapReduceJob/jobs'):
            if node == 'x':
                continue
            if node == 'j':
                for node2 in os.listdir('/usr/share/statbox-class-filestorage/Action-MapReduceJob/jobs/j'):
                    os.symlink(
                        os.path.join('/usr/share/statbox-class-filestorage/Action-MapReduceJob/jobs/j', node2),
                        os.path.join(extract_dir, 'j', node2)
                    )
                continue
            os.symlink(
                os.path.join('/usr/share/statbox-class-filestorage/Action-MapReduceJob/jobs', node),
                os.path.join(extract_dir, node)
            )

        with sdk2.helpers.ProcessLog(self, logger='chmod') as pl:
            return_code = sp.Popen(
                ['chmod', '-R', '777', os.path.dirname(self.Context.statbox_class_filestorage_path)],
                shell=False, stdout=pl.stdout, stderr=sp.STDOUT
            ).wait()
            if return_code != 0:
                logging.error('Failed to CHMOD')

    @track
    def _prepare_configs_repos(self):
        extract_dir = os.path.join(self.Context.statbox_class_filestorage_path, 'Users-Configs')
        tools.mkdir_p(extract_dir)
        configs_resource = self.Parameters.config_repos
        repos_resource_data = sdk2.ResourceData(configs_resource)
        repos_resource_path = repos_resource_data.path
        with sdk2.helpers.ProcessLog(self, logger='tar_configs') as pl:
            return_code = sp.Popen(
                ['tar', '-xvf', str(repos_resource_path), '-C', extract_dir],
                shell=False, stdout=pl.stdout, stderr=sp.STDOUT
            ).wait()
            if return_code != 0:
                raise Exception('Failed to unpack configs repos')

    @tools.silently
    @track
    def _prepare_certs(self):
        for cert_name in ('api-reg-ru.pem', 'tutbaikin.pem'):
            cert_body = sdk2.Vault.data('STATINFRA', cert_name)
            tools.mkdir_p('/home/sandbox/certs-partner-interface')
            with open('/home/sandbox/certs-partner-interface/%s' % cert_name, 'w') as f:
                f.write(cert_body)

    def _get_secrets(self, statobj_id, secrets_names):
        def _vault_data(owner, name):
            try:
                return sdk2.Vault.data(owner, name)
            except VaultNotFound:
                logging.warning('SECRETS: There is no secret {} with owner {}'.format(name, owner))
            except VaultNotAllowed:
                logging.warning('SECRETS: Vault "{}" permission denied'.format(name))

        def _set_secret(owner, name, secret):
            os.environ['{}__{}__{}'.format(SECRETS_ENV_PREFIX, owner.upper(), name.upper())] = secret

        def _valid_secret(secret, name):
            if secret:
                if not hasattr(secret, 'description'):
                    logging.warning('SECRETS: Something strange with secret {}'.format(name))
                    return False
                if not isinstance(secret.description, six.string_types):
                    logging.warning('SECRETS: Seems "{}" description is not a string'.format(name))
                    return False
                if len(secret.description) == 0:
                    logging.warning('SECRETS: Secret\'s "{}" description can\'t be empty'.format(name))
                    return False
            else:
                return False
            return True

        def _allow_secret(secret, name):
            allowed_list = secret.description.split()
            for allowed in allowed_list:
                if allowed.endswith('/'):
                    if statobj_id.startswith(allowed):
                        return True
                else:
                    if statobj_id == allowed:
                        return True
            logging.warning('SECRETS: Secret "{}" isn\'t allowed to use in this action'.format(name))
            return False

        if not secrets_names:
            return
        for owner in secrets_names:
            for name in secrets_names[owner]:
                secret = _vault_data(owner, name)
                if _valid_secret(secret, name) and _allow_secret(secret, name):
                    _set_secret(owner, name, secret)

    @track
    def _get_action_secrets(self):
        self._get_secrets(
            statobj_id=self.Context.action_info['action_id'],
            secrets_names=self.Context.action_info['secrets_names'],
        )

    @track
    def _get_job_secrets(self):
        if 'job' not in self.Context.job_info and 'meta' not in self.Context.job_info:
            logging.info('There are no job or meta field in job_info')
            return
        self._get_secrets(
            statobj_id=self.Context.job_info['job'],
            secrets_names=self.Context.job_info['meta'].get('secrets_names', {}),
        )

    def _action_to_path(self, action_type, action_id):
        if action_type == 'external':
            return os.path.join(
                self.Context.statbox_class_filestorage_path,
                'ActionDatabase',
                'external',
                action_id[9:],
            )
        elif action_type == 'arcadia':
            return os.path.join(self.Context.statbox_class_filestorage_path, 'ActionDatabase', action_id)

    def _exec_file_name(self, dir):
        if self.Context.action_info.get('exec_file_name') is not None:
            name = self.Context.action_info['exec_file_name']
            if os.path.isfile(os.path.join(dir, name)):
                return name
            else:
                return None
        for name in ['job', 'action']:
            if os.path.isfile(os.path.join(dir, name)):
                return name
        return None

    @track
    def _execute_action(self, resource_paths):
        log_dir = sandboxsdk.paths.get_logs_folder()
        process_env = os.environ.copy()
        process_env['STATBOX_LOG_PATH'] = log_dir
        process_env['STATBOX_RUN_ID'] = str(self.id)
        process_env['STATBOX_TASK_ID'] = str(
            getattr(self.Parameters.primary_sandbox_task_id, 'id', self.Parameters.primary_sandbox_task_id)
        )
        process_env['STATBOX_EXEC_UUID'] = 'DEPRECATED'
        process_env['STATBOX_YT_USER'] = 'statbox'
        process_env['STEP_EVENTS'] = json.dumps(self.Parameters.step_events or [])
        process_env['STEP_EVENT_DEPS_CASE'] = self.Parameters.event_deps_case
        process_env['REQUESTS_CA_BUNDLE'] = '/etc/ssl/certs/ca-certificates.crt'
        process_env['PERL_LWP_SSL_CA_PATH'] = '/etc/ssl/certs/'
        process_env['STATBOX_CLASS_CONF'] = self.Context.statbox_class_filestorage_path
        process_env['STATBOX_CONFIG_PATH'] = os.path.join(self.Context.statbox_class_filestorage_path, 'Users-Configs')
        process_env['STATBOX_ENVIRONMENT_SPEC'] = config.si_env
        process_env['STATBOX_ACTION_ID'] = self.Parameters.action_id
        process_env['EXTRA_RESOURCE_PATHS'] = json.dumps(resource_paths)
        process_env['PGSSLROOTCERT'] = '/etc/ssl/certs/YandexInternalRootCA.pem'

        # e.g. to release semaphores in the middle of the task
        # https://st.yandex-team.ru/STATINFRA-9686
        process_env['SANDBOX_TASK_TOKEN'] = common.rest.Client._external_auth.task_token

        pythonpath_backup = os.environ['PYTHONPATH']
        process_env['PYTHONPATH'] = self.Context.pythonpath

        logging.info('PROCESS ENVIRONMENT: {}'.format(
            {
                i: process_env[i]
                for i in process_env
                if (not i.startswith('{}__'.format(SECRETS_ENV_PREFIX)) and i != 'SANDBOX_TASK_TOKEN')
            }
        ))

        action_id = self.Parameters.action_id
        action_params = ' '.join(
            '%s=%s' % (k, json.dumps(v)) for k, v in self.Parameters.event_params.items()
        ) if self.Parameters.event_params is not None else ''

        exit_code = 0
        out = None
        failed = False
        self.tcp_dump_proc = tools.start_tcpdump(ports=[80, 443, 8123])
        with sdk2.helpers.ProcessLog(self, logger='exec') as pl:
            pl.logger.propagate = 1
            try:
                if self.Context.action_info['action_type'] in ['external', 'arcadia']:
                    action_dir_path = self._action_to_path(
                        self.Context.action_info['action_type'],
                        self.Parameters.action_id,
                    )
                    tmpdir = str(tempfile.mkdtemp())
                    action_tmp_dir = os.path.join(tmpdir, 'action')
                    shutil.copytree(action_dir_path, action_tmp_dir)
                    executable_file = self._exec_file_name(action_tmp_dir)
                    if executable_file is None:
                        raise OSError('executable file wasn`t found')
                    # give execute priveleges and run exec file
                    action_execution_cmd = 'cd {atdir} && chmod +x {execname} && ./{execname} {params}'.format(
                        atdir=action_tmp_dir,
                        execname=executable_file,
                        params=action_params,
                    )

                    proc = sp.Popen(
                        action_execution_cmd,
                        shell=True,
                        stdout=sp.PIPE,
                        stderr=sp.STDOUT,
                        env=process_env,
                    )

                    if action_id == "external:traf-action/metrika/searches":
                        # DEBUG SPIKE
                        time.sleep(300)

                    lines = []
                    with open(os.path.join(log_dir, ACTION_STDOUTERR_FILENAME), 'wb') as out_file:
                        for line in iter(proc.stdout.readline, ''):
                            lines.append(line)
                            out_file.write(hide_oauth(line))
                            out_file.flush()
                    proc.stdout.close()
                    returncode = proc.wait()
                    out = ''.join(lines)
                    if returncode != 0:
                        raise sp.CalledProcessError(
                            returncode=returncode,
                            cmd=action_execution_cmd,
                            output=out,
                        )
                else:
                    out = sp.check_output(
                        ['/usr/bin/statbox-ctl.pl execute-action %s %s' % (action_id, action_params)],
                        shell=True, stderr=sp.STDOUT, env=process_env
                    )
            except sp.CalledProcessError as e:
                pl.logger.error(hide_oauth(e.output))
                self.set_info('Action process return code: {}'.format(e.returncode))
                self.set_info('ERROR: %s' % hide_oauth(e.output))
                exit_code = e.returncode
                out = e.output
                failed = True
            except OSError as e:
                pl.logger.error(e.message)
                self.set_info("Executable file was not found")
                failed = True
                exit_code = 1
                out = e.message
            finally:
                os.environ['PYTHONPATH'] = pythonpath_backup
                tools.stop_tcpdump(self.tcp_dump_proc)

            pl.logger.info(out)

        out = six.text_type(out, errors='ignore')
        return exit_code, out, failed

    @track
    def _store_python_statinfra_log_position(self):
        python_log_end_pos = 0
        if os.path.exists(config.PYTHON_LOG_PATH):
            with open(config.PYTHON_LOG_PATH) as f:
                f.seek(0, os.SEEK_END)
                python_log_end_pos = f.tell()
            logging.info(
                '%s before executing counts: %s',
                config.PYTHON_LOG_PATH, sp.check_output(['wc', config.PYTHON_LOG_PATH])
            )
        logging.info('python-statinfra.log position is {}'.format(python_log_end_pos))
        self.Context.python_statinfra_log_position = python_log_end_pos

    @track
    def _copy_python_statinfra_log(self):
        if os.path.exists(config.PYTHON_LOG_PATH):
            logging.info(
                '%s after executing counts: %s',
                config.PYTHON_LOG_PATH, sp.check_output(['wc', config.PYTHON_LOG_PATH])
            )
            log_path = os.path.join(sandboxsdk.paths.get_logs_folder(), 'python-statinfra.log')
            task_id = getattr(self.Parameters.primary_sandbox_task_id, 'id', self.Parameters.primary_sandbox_task_id)
            with open(config.PYTHON_LOG_PATH) as f_in, open(log_path, 'w') as f_out:
                f_in.seek(self.Context.python_statinfra_log_position or 0)
                logging.info('Copy {} to {} from start position {}, with filter task_id={}'.format(
                    config.PYTHON_LOG_PATH,
                    log_path,
                    self.Context.python_statinfra_log_position or 0,
                    task_id)
                )
                f_out.writelines(
                    filter(
                        lambda l: '\ttask_id={}\t'.format(task_id) in l,
                        f_in.readlines()
                    )
                )
                f_out.write('\n')
            logging.info('{} counts: {}'.format(log_path, sp.check_output(['wc', log_path])))
        else:
            logging.info("Path {} doesn't exist".format(config.PYTHON_LOG_PATH))

    @track
    def _register_logs(self):
        registry = common.config.Registry()
        if registry.common.installation != ctm.Installation.LOCAL:
            for log_name in config.LOGS_TO_REGISTER:
                log_path = os.path.join(sandboxsdk.paths.get_logs_folder(), log_name)
                log_type = sandbox.agentr.types.LogType.STAT_TASK

                if log_name == 'statbox-daemon-sandbox-tskv.log':
                    log_type = sandbox.agentr.types.LogType.STATBOX_DAEMON
                if log_name == 'task-execution-log.log':
                    log_type = sandbox.agentr.types.LogType.STAT_TASK_EXECUTION
                if log_name == 'python-statinfra.log':
                    log_type = sandbox.agentr.types.LogType.PYTHON_STATINFRA

                self.agentr.register_log(log_path, log_type=log_type)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        self.Context.start_time = datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        self._check_restart()
        if not self.Context.action_info:
            self._update_action_info()
        if self.is_copy_table():
            job_id = None
            if 'for_job' in self.Parameters.event_params:
                job_id = self.Parameters.event_params['for_job']
            self._update_job_info(job_id=job_id)
        self._set_hints()
        self._prepare_libs()
        self._prepare_venvs()
        self._prepare_repos()
        self._register_logs()
        self._prepare_certs()
        self._get_action_secrets()
        if self.is_job_runner() or self.is_copy_table():
            self._get_job_secrets()
        self._store_python_statinfra_log_position()
        resource_paths = self._prepare_extra_resources()

        exit_code, stdouterr, failed = self._execute_action(resource_paths)

        write_task_execution_log(self, exit_code=exit_code, stdouterr=stdouterr, failed=failed)
        self._copy_python_statinfra_log()

        send_notifications(
            parameters=self.Parameters,
            failed=failed,
            stdouterr=hide_oauth(stdouterr),
            action_info=self.Context.action_info,
            start_time=self.Context.start_time,
            task_id=self.id
        )

        restart = failed
        if self.is_job_runner():
            restart = restart and TaskInspector(self, stdouterr).run() == RESTART_TASK

        if (restart or is_cyclic_action(self.Context.action_info, stdouterr)) and \
                not action_deleted_from_repo(stdouterr, self.Parameters.action_id):
            restart_action(
                self,
                stdouterr=stdouterr,
                failed=failed
            )

        if failed:
            # raise StatinfraTaskFailure(stdouterr)
            raise StatinfraTaskFailure()

    @track
    def _set_hints(self):
        hints = self.Context.action_info.get('custom_params', {}).get('sandbox_hints', [])
        for hint in hints:
            if hint in self.Parameters.event_params:
                value = self.Parameters.event_params[hint]
                self.set_info('Set hint {}={}'.format(hint, value))
                self.hint('{}={}'.format(hint, value))
            else:
                self.set_info('WARNING: missing hint {}'.format(hint))

    @track
    def _check_restart(self):
        if self.agentr.iteration > 1 and self.Context.executing:
            # check iteration so if the task is cloned it would not go waiting
            logging.info('Noticed that the task was restarted - waiting default timeout')
            self.Context.executing = False
            raise sdk2.WaitTime(15 * 60)  # 15 minutes timeout on TEMPORARY
        self.Context.executing = True

    def on_break(self, prev_status, status):
        logging.debug('func on_break was called')
        write_task_execution_log(self, failed=True)
        send_notifications(
            parameters=self.Parameters,
            failed=True,
            stdouterr='Task has been stopped',
            action_info=self.Context.action_info,
            start_time=self.Context.start_time,
            task_id=self.id
        )

    def on_terminate(self):
        logging.info('func on_terminate was called')
        self._copy_python_statinfra_log()

    def on_finish(self, prev_status, status):
        solomon_push()

    @property
    def footer(self):
        prev_link = '<a href="' + '{}/task/{}/view'.format(common.utils.server_url(), self.Context.copy_of) + \
                    '">%s</a>' % self.Context.copy_of
        next_link = '<a href="' + '{}/task/{}/view'.format(common.utils.server_url(), self.Context.new_task_id) + \
                    '">%s</a>' % self.Context.new_task_id
        return '''
        Previous: {prev}<br>
        Next: {next}
        '''.format(prev=prev_link, next=next_link)
