import os

from sandbox import sdk2
from sandbox import sandboxsdk
from sandbox.sandboxsdk import ssh
from sandbox.sdk2.helpers import subprocess as sp


CONFIG_TEMPLATE = '''
[distutils]
index-servers =
    yandex

[yandex]
repository: https://pypi.yandex-team.ru/simple/
username: {access_key}
password: {secret_key}

'''


class UploadToPypi(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 10000
        cores = 1
        environments = (
            sandboxsdk.environments.PipEnvironment('GitPython'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        git_url = sdk2.parameters.Url('Git url')
        git_tag = sdk2.parameters.String('Git tag')

        vault_user = sdk2.parameters.String('Vault owner', default='STATINFRA')
        vault_access_key_record = sdk2.parameters.String(
            'Pypi access-key record in vault',
            default='robot-statinfra-pypi-access',
        )
        vault_secret_key_record = sdk2.parameters.String(
            'Pypi secret-key record in vault',
            default='robot-statinfra-pypi-secret',
        )

    class PutVaultDataToHome(object):
        def __init__(self, access_key, secret_key):
            self.restore_original_pypirc = False
            self.original_pypirc_backup = None
            self.config_path = os.path.expanduser('~/.pypirc')

            self.access_key = access_key
            self.secret_key = secret_key

        def __enter__(self):
            if os.path.exists(self.config_path):
                self.restore_original_pypirc = True
                self.original_pypirc_backup = self.config_path + '_backup'
                os.rename(self.config_path, self.original_pypirc_backup)

            with open(self.config_path, 'w') as f:
                f.write(
                    CONFIG_TEMPLATE.format(
                        access_key=self.access_key,
                        secret_key=self.secret_key
                    )
                )

        def __exit__(self, *args):
            os.remove(self.config_path)
            if self.restore_original_pypirc:
                os.rename(self.original_pypirc_backup, self.config_path)

    def on_execute(self):
        import git

        with ssh.Key(self, 'STATINFRA', 'robot-statinfra-ssh-key'):
            git.Git().clone(self.Parameters.git_url, depth=1)
        path = self.Parameters.git_url.split('/')[-1].split('.')[0]
        g = git.Git(path)
        g.checkout(self.Parameters.git_tag)
        os.chdir(path)

        access_key = sdk2.Vault.data(self.Parameters.vault_user, self.Parameters.vault_access_key_record)
        secret_key = sdk2.Vault.data(self.Parameters.vault_user, self.Parameters.vault_secret_key_record)
        with UploadToPypi.PutVaultDataToHome(access_key=access_key, secret_key=secret_key):
            with sdk2.helpers.ProcessLog(self, logger='exec') as pl:
                sp.check_call(
                    ['python', 'setup.py', 'sdist', 'upload', '-r', 'yandex'],
                    shell=False, stdout=pl.stdout, stderr=sp.STDOUT
                )

        task_class = sdk2.Task['BUILD_PY_PACKAGE']
        subtask = task_class(
            self,
            description='Python wheel of %s' % path,
            create_sub_task=True,
            package_name='%s==%s' % (path, self.Parameters.git_tag),
        ).enqueue()

        self.set_info('BUILD_PY_PACKAGE started: %s' % subtask.id)
