# coding=utf-8
from __future__ import unicode_literals

import logging
import shutil
import os
import json

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.common.build.YaMake2 import YaMake2

from pathlib2 import Path


LOGGER = logging.getLogger(__name__)


class CalculationResource(sdk2.Resource):
    pass


class ResourceBuilder(object):
    def __init__(self):
        self.description = "No description"
        self.targets = {}
        self.files = {}

    def add_description(self, description):
        self.description = description
        return self

    def add_target(self, path_in_resource, target):
        self.targets[path_in_resource] = target
        return self

    def add_file(self, path_in_resource, data):
        self.files[path_in_resource] = data
        return self


class BuildCalculationResource(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 8000
        cores = 1
        ram = 4 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.String('Arcadia url', required=True, description="e.g. arcadia-arc:/#trunk")
        targets = sdk2.parameters.Dict('Targets')

        with sdk2.parameters.RadioGroup('Type') as type:
            type.values['nile_over_yql'] = type.Value(value='nile_over_yql')
            type.values['nile_over_yt'] = type.Value(value='nile_over_yt')
            type.values['runplan'] = type.Value(value='runplan')

        meta = sdk2.parameters.JSON('Meta', required=True, default={})
        data = sdk2.parameters.Dict('Data')
        resource_description = sdk2.parameters.String('Resource Description', required=True)

        with sdk2.parameters.Output:
            release_resource = sdk2.parameters.Resource(
                "Calculation Resource",
                resource_type=CalculationResource
            )

    @property
    def plan(self):
        if hasattr(self, '_plan'):
            return self._plan

        meta = self.Parameters.meta.copy()
        meta['type'] = self.Parameters.type

        builder = ResourceBuilder() \
            .add_description(self.Parameters.resource_description) \
            .add_file("meta.json", json.dumps(meta, indent=4))

        for path, target in self.Parameters.targets.items():
            builder.add_target(path, target)

        for path, content in self.Parameters.data.items():
            builder.add_file(path, content)

        self._plan = builder
        return builder

    def create_yamake_subtask(self, target):
        subtask = YaMake2(
            self,
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            targets=target,
            use_arc_instead_of_aapi=True
        ).enqueue()

        return subtask.id

    def copy_built_targets(self, target, destination):
        subtask = self.find(YaMake2, input_parameters={'targets': target}).first()

        resource = sdk2.Resource.find(
            sdk2.Resource['BUILD_OUTPUT'],
            task=subtask
        ).first()

        source = sdk2.ResourceData(resource).path / target
        files = list(source.glob('*'))
        if len(files) == 1:  # assume one file in dir
            source = source / files[0]
        else:
            raise RuntimeError('more than one file in result of YA_MAKE_2 task')

        LOGGER.info('Copy %s to %s', source, destination)
        shutil.copy(source.as_posix(), destination.as_posix())
        os.chmod(destination.as_posix(), 0o755)

    def prepare_resource(self):
        resource = CalculationResource(self, self.plan.description, 'calc')

        data = sdk2.ResourceData(resource)

        for path, target in self.plan.targets.items():
            path = Path(data.path) / path
            path.parent.mkdir(parents=True, exist_ok=True)
            self.copy_built_targets(target, path)

        for path, content in self.plan.files.items():
            path = Path(data.path) / path
            path.parent.mkdir(parents=True, exist_ok=True)
            LOGGER.info('Create file %s', path)
            with path.open('wb') as file:
                file.write(content)

        data.ready()

        return resource

    def on_execute(self):
        with self.memoize_stage.create_resource(commit_on_entrance=False):
            raise sdk2.WaitTask(
                [self.create_yamake_subtask(target)
                 for target in self.plan.targets.values()],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

        self.Parameters.release_resource = self.prepare_resource()
