# coding=utf-8
from __future__ import unicode_literals

import logging
import shutil
from sandbox import sdk2

import sandbox.common.types.task as ctt
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.projects.statkey.BuildCalculationResource import CalculationResource

LOGGER = logging.getLogger(__name__)

SECRET, SECRET_KEY = 'sec-01czmzxrk51tsncdgt28k6ems5', 'ROBOT_STATKEY_OAUTH'


class StatkeyReleaseResource(CalculationResource):
    releasable = True
    releasers = ['komandr']


REPO_CONFIG = '''
---
_default:
  templates:
    root: statbox
    home: $root/home/$user
    job_prefix: x/statkey-job
    job_root: $root/statbox/$scale/$job_prefix/$job_name
    tmp: //tmp/statbox/$user/$job_prefix/$scale/$job_name
    heavy_dict_root: $root/heavy_dict
    heavy_result_root: $root/heavy_result
  sentry_team: statkey
  responsible: ['robot-statkey@yandex-team.ru', ]
  token:
    owner: STATKEY
    name: ROBOT_STATKEY_OAUTH
  secrets_names:
    STATKEY:
    - ROBOT_STATKEY_OAUTH
  porto_layers: ['//statbox/resources/porto_layer_libs.tar.gz', '//statbox/resources/porto_layer2.tar.gz', '//porto_layers/base/precise/porto_layer_search_ubuntu_precise_app_lastest.tar.gz']
  use_symlinks: 1
  allow_inherit_clusters: 0
  yt_pool: statbox-cooked-logs-batch
  allow_yt_cloud: 0
  weight: 90
  max_speculative_job_count_per_task: 50
'''  # noqa


class StatkeyBuildReleaseResource(sdk2.Task):
    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group('Секреты') as secrets:
            secret = sdk2.parameters.YavSecret('Секрет с OAUTH-токеном', default=SECRET)
            secret_key = sdk2.parameters.String('Ключ в секрете, где лежит OAUTH-токен', default=SECRET_KEY)

        with sdk2.parameters.Group('Бинарники') as binary:
            run_python_udf = sdk2.parameters.String(
                'Путь до ya.make run_python_udf', default='yql/tools/run_python_udf', required=True)
            statkey_job = sdk2.parameters.String(
                'Путь до расчёта',
                description='Например: statbox/statkey/jam/jobs/cubes/bs_money/v2.1', required=True)

        with sdk2.parameters.Output:
            release_resource = sdk2.parameters.Resource(
                "Statkey Release Resource",
                resource_type=StatkeyReleaseResource
            )

    def create_yamake_subtask(self, target):
        subtask = YaMake2(
            self,
            checkout_arcadia_from_url='arcadia-arc:/#trunk',
            targets=target,
            use_arc_instead_of_aapi=True
        ).enqueue()

        return subtask.id

    def copy_built_targets(self, target, destination):
        subtask = self.find(YaMake2, input_parameters={'targets': target}).first()

        resource = sdk2.Resource.find(
            sdk2.Resource['BUILD_OUTPUT'],
            task=subtask
        ).first()

        source = sdk2.ResourceData(resource).path / target
        source = source / next(source.glob('*'))  # assume one file in dir

        LOGGER.info('Copy %s to %s', source, destination)
        shutil.copy(source.as_posix(), destination.as_posix())

    def copy_repo_config(self, destination):
        with destination.open('w') as file:
            file.write(REPO_CONFIG)

    def prepare_release_resource(self):
        resource = StatkeyReleaseResource(self, "Built {}".format(self.Parameters.statkey_job), "release")
        data = sdk2.ResourceData(resource)
        path = data.path / 'bin'
        path.mkdir(0o755, parents=True, exist_ok=True)

        self.copy_built_targets(self.Parameters.run_python_udf, path / 'run_python_udf')
        self.copy_built_targets(self.Parameters.statkey_job, path / 'statkey_job')
        self.copy_repo_config(data.path / '.repo_config.tyaml')

        data.ready()

        return resource

    def on_execute(self):
        super(StatkeyBuildReleaseResource, self).on_execute()

        with self.memoize_stage.statkey_create_resource(commit_on_entrance=False):
            raise sdk2.WaitTask(
                [self.create_yamake_subtask(self.Parameters.run_python_udf),
                 self.create_yamake_subtask(self.Parameters.statkey_job)],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

        self.Parameters.release_resource = self.prepare_release_resource()
