# coding=utf-8
from __future__ import unicode_literals

import logging

import json

from sandbox import sdk2
from sandbox.projects.statkey.BuildCalculationResource import CalculationResource
from sandbox.sdk2.helpers import gdb


SECRET, SECRET_KEY = 'sec-01czmzxrk51tsncdgt28k6ems5', 'ROBOT_STATKEY_OAUTH'


class StatkeyRunNile(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 8000
        cores = 1
        # dns = ctm.DnsType.DNS64  # copied from STATINFRA_TASK
        ram = 8 * 1024
        # ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 2048, None)  # copied from STATINFRA_TASK

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group('Секреты') as secrets_group:
            secret = sdk2.parameters.YavSecret('Секрет с OAUTH-токеном', description='e.g. ' + SECRET)
            secret_key = sdk2.parameters.String('Ключ в секрете, где лежит OAUTH-токен', description='e.g.' + SECRET_KEY)

        with sdk2.parameters.RadioGroup('Режим ресурса') as resource_mode:  # TODO deprecated, remove
            resource_mode.values['resource-nyql-tier0'] = resource_mode.Value(value='Resource (nile over yql, tier0)', default=True)
            resource_mode.values['build-nyql-tier0'] = resource_mode.Value(value='Build from trunk (nile over yql, tier0)')

        with resource_mode.value['resource-nyql-tier0']:
            release_resource = sdk2.parameters.Resource('Реcурс с релизом', resource_type=CalculationResource, required=True)

        with resource_mode.value['build-nyql-tier0']:
            statkey_job = sdk2.parameters.String('Путь до расчёта', required=True)

        with sdk2.parameters.RadioGroup('Режим') as mode:
            mode.values['run'] = mode.Value(value='run', default=True)
            mode.values['info'] = mode.Value(value='info')

        with sdk2.parameters.RadioGroup('scale') as scale:
            scale.values['daily'] = scale.Value(value='daily', default=True)
            scale.values['weekly'] = scale.Value(value='weekly')
            scale.values['monthly'] = scale.Value(value='monthly')

        dates = sdk2.parameters.String('dates', required=True)

        with sdk2.parameters.Output:
            info = sdk2.parameters.JSON('Результат ручки info')

    @property
    def resource_path(self):
        if hasattr(self, '_resource_path'):
            return self._resource_path

        if self.Parameters.resource_mode != 'resource-nyql-tier0':
            raise DeprecationWarning

        self._resource_path = sdk2.ResourceData(self.Parameters.release_resource).path
        return self._resource_path

    @property
    def meta(self):
        if hasattr(self, '_meta'):
            return self._meta

        try:
            with (self.resource_path / 'meta.json').open() as file:
                self._meta = json.load(file)
        except IOError:
            self._meta = {"type": None}

        return self._meta

    def on_execute(self):
        super(StatkeyRunNile, self).on_execute()

        if self.Parameters.resource_mode != 'resource-nyql-tier0':
            raise DeprecationWarning

        if self.meta['type'] is None:
            executor = NileOverYQLExecutorDeprecated(self)
        elif self.meta['type'] == 'nile_over_yql':
            executor = NileOverYQLExecutor(self)
        elif self.meta['type'] == 'nile_over_yt':
            executor = NileOverYTExecutor(self)
        elif self.meta['type'] == 'custom':
            raise NotImplementedError

        if self.Parameters.mode == 'run':
            execute = executor.run
        elif self.Parameters.mode == 'info':
            execute = executor.info

        execute(self.Parameters.scale, self.Parameters.dates)


class ExecutorBase(object):
    def __init__(self, task):
        logging.info("Initialize {}", type(self))
        self._task = task

    def _execute(self, argv, env, return_output):
        with sdk2.helpers.ProcessLog(self._task, logger='run-binary') as log:
            for stream in ['out', 'err']:
                self._task.set_info(
                    gdb.get_html_view_for_logs_file(
                        'Std' + stream, getattr(log, 'std' + stream).path.relative_to(self._task.log_path()), self._task.log_resource
                    ),
                    do_escape=False
                )

            if return_output:
                return sdk2.helpers.subprocess.check_output(argv, env=env, stderr=log.stderr)
            else:
                sdk2.helpers.subprocess.check_call(argv, env=env, stderr=log.stderr, stdout=log.stdout)
                return

    def run(self, scale, timestamp):  # TODO проброска параметров через Реактор
        self._execute(self.argv_run(scale, timestamp),
                      self.env_run(),
                      return_output=False)

    def info(self, scale, timestamp):
        output = self._execute(self.argv_info(scale, timestamp),
                               self.env_info(),
                               return_output=True)

        self._task.Parameters.info = json.loads(output)


class NileOverYQLExecutorDeprecated(ExecutorBase):
    def argv_run(self, scale, timestamp):
        return [
            (self._task.resource_path / 'bin/run_python_udf').as_posix(),
            (self._task.resource_path / 'bin/statkey_job').as_posix(),
            'run',
            '--proxy', 'hahn',
            '--dates', timestamp,
            '--scale', scale,
            '--use-yql'
        ]

    def env_run(self):
        oauth = self._task.Parameters.secret.data()[self._task.Parameters.secret_key]
        return {'YT_TOKEN': oauth, 'YQL_TOKEN': oauth}

    def argv_info(self, scale, timestamp):
        return [
            (self._task.resource_path / 'bin/run_python_udf').as_posix(),
            (self._task.resource_path / 'bin/statkey_job').as_posix(),
            'info',
            '--proxy', self._task.meta['cluster'],
            '--dates', timestamp,
            '--scale', scale,
            '--use-yql'
        ]

    def env_info(self):
        return {}


class NileOverYQLExecutor(ExecutorBase):
    def argv_run(self, scale, timestamp):
        return [
            (self._task.resource_path / 'bin/run_python_udf').as_posix(),
            (self._task.resource_path / 'bin/udf.so').as_posix(),
            'run',
            '--proxy', self._task.meta['cluster'],
            '--dates', timestamp,
            '--scale', scale
        ]

    def env_run(self):
        oauth = self._task.Parameters.secret.data()[self._task.Parameters.secret_key]
        return {'YT_TOKEN': oauth, 'YQL_TOKEN': oauth}

    def argv_info(self, scale, timestamp):
        return [
            (self._task.resource_path / 'bin/run_python_udf').as_posix(),
            (self._task.resource_path / 'bin/udf.so').as_posix(),
            'info',
            '--proxy', self._task.meta['cluster'],
            '--dates', timestamp,
            '--scale', scale
        ]

    def env_info(self):
        oauth = self._task.Parameters.secret.data()[self._task.Parameters.secret_key]
        return {'YT_TOKEN': oauth, 'YQL_TOKEN': oauth}


class NileOverYTExecutor(ExecutorBase):
    def argv_run(self, scale, timestamp):
        return [
            (self._task.resource_path / 'bin/job').as_posix(),
            'run',
            '--proxy', self._task.meta['cluster'],
            '--dates', timestamp,
            '--scale', scale
        ]

    def env_run(self):
        oauth = self._task.Parameters.secret.data()[self._task.Parameters.secret_key]
        return {'YT_TOKEN': oauth}

    def argv_info(self, scale, timestamp):
        return [
            (self._task.resource_path / 'bin/job').as_posix(),
            'info',
            '--proxy', self._task.meta['cluster'],
            '--dates', timestamp,
            '--scale', scale
        ]

    def env_info(self):
        oauth = self._task.Parameters.secret.data()[self._task.Parameters.secret_key]
        return {'YT_TOKEN': oauth}
