import logging
import requests
import json

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import task_env


DC_BASE_PROD_URL = "https://datacatalog.in.yandex-team.ru/api/v1"
DC_BASE_PREPROD_URL = "https://preprod.datacatalog.in.yandex-team.ru/api/v1"

TVM_SRC = 2035845
TVM_DST = 2033503


class DataCatalogNotifyDeadlineEvents(sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        cores = 1
        ram = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        deadline_date = sdk2.parameters.String("Deadline date (YYYY-MM-DD)", default=None)
        is_prod = sdk2.parameters.Bool("Is production?", default=True)

    @staticmethod
    def get_service_ticket():
        import tvmauth

        tvm_client = tvmauth.TvmClient(
            tvmauth.TvmApiClientSettings(
                self_tvm_id=TVM_SRC,
                enable_service_ticket_checking=True,
                enable_user_ticket_checking=tvmauth.BlackboxEnv.Test,
                self_secret=sdk2.yav.Secret("sec-01g6qz0zzrtcvka7nqh8z2gwy0").data()["client_secret"],
                dsts={"DC": TVM_DST},
            ),
        )
        return tvm_client.get_service_ticket_for("DC")

    def post_request(self, url, payload):
        retry_strategy = Retry(
            total=3,
            status_forcelist=[429, 500, 502, 503, 504],
            method_whitelist=["POST"],
            backoff_factor=2,
        )
        adapter = HTTPAdapter(max_retries=retry_strategy)
        session = requests.Session()
        session.mount("https://", adapter)

        logging.info("Start request to %s", url)
        service_ticket = self.get_service_ticket()
        req = session.post(
            url=url,
            json=payload,
            headers={
                "X-Ya-Service-Ticket": service_ticket,
                "Content-Type": "application/json",
                "Accept": "application/json",
            },
        )

        logging.info("Finish request to %s with code %i", url, req.status_code)
        logging.info("Response:\n%s", json.dumps(req.json(), sort_keys=False, indent=4))

        if req.status_code == 200:
            return

        raise TaskFailure("Request to %s failed", url)

    def notify_deadline_events(self):
        import arrow

        dc_base_url = DC_BASE_PROD_URL if self.Parameters.is_prod else DC_BASE_PREPROD_URL
        event_notification_endpoint = dc_base_url + "/event/notify"
        arrow_datetime = arrow.get(str(self.Parameters.deadline_date), tzinfo="Europe/Moscow")
        payload = dict(
            startDeadlineTime=arrow_datetime.int_timestamp,
            endDeadlineTime=arrow_datetime.shift(days=1).int_timestamp - 1,
        )

        self.post_request(event_notification_endpoint, payload)

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={"name": "DataCatalogNotifyDeadlineEvents", "release": ctt.ReleaseStatus.STABLE},
        ).first()

    def on_execute(self):
        self.notify_deadline_events()
