# coding=utf-8

import logging
from sandbox import sdk2
import sandbox.common.types.task as ctt

YQL_TOKEN_OWNER = 'STATKEY'
YQL_TOKEN_NAME = 'YQL_TOKEN'


class MoneyMapDatalens(sdk2.Task):

    class Parameters(sdk2.Parameters):
        yt_cluster = sdk2.parameters.String('Yt cluster', default='hahn')
        dates = sdk2.parameters.List("Dates")
        dl_table_prefix = sdk2.parameters.String('DL table prefix', default='//home/statkey/moneymap/datalens/')
        dwh_table_prefix = sdk2.parameters.String('DWH table prefix', default='//statbox/cube/daily/dwh_200/v3/report/1_phase')
        revenue_table_prefix = sdk2.parameters.String('Revenue table prefix', default='//home/statkey/moneymap/dev/STATKEY-3492/dwh_grouped/')

    class Slice():
        def __init__(self, order_type, product):
            self.order_type = order_type
            self.prouct = product

        def get_slice_name(self):
            return "{}$$${}".format(self.order_type, self.prouct)

        def __eq__(self, other):
            if not isinstance(other, type(self)):
                # don't attempt to compare against unrelated types
                return NotImplemented

            return self.order_type == other.order_type and self.prouct == other.prouct

        def __hash__(self):
            return hash((self.order_type, self.prouct))

    class AbcEntity():
        def __init__(self, id, name, parent_id=None, revenue_dict=None, is_oebs=False):
            self.id = id
            self.name = name
            self.children = []
            self.parent = None
            self.parent_id = parent_id
            self.revenue_dict = revenue_dict if revenue_dict is not None else {}
            self.is_oebs = is_oebs
            self.is_vs = False
            self.vs = None

        def add_child(self, child):
            child.parent = self
            self.children.append(child)

        def set_revenue(self, slice, value):
            if slice in self.revenue_dict:
                self.revenue_dict[slice] += value
            else:
                self.revenue_dict[slice] = value

        def get_revenue(self, by_slice=None):
            sum = 0.0
            for name, revenue in self.revenue_dict.items():
                if by_slice is None or name == by_slice:
                    sum = sum + revenue
            return sum

        def setVsForChild(self, vs):
            for child in self.children:
                child.vs = vs
                child.setVsForChild(vs)

        def calc_child(self, with_self=False, by_slice=None):
            if with_self or len(self.children) == 0:
                sum = self.get_revenue(by_slice)
            else:
                sum = 0.0

            for child in self.children:
                sum = sum + child.calc_child(with_self=True, by_slice=by_slice)
            return sum

        def get_hierarchical_name(self):
            if self.parent is None:
                return self.name
            else:
                parent_name = self.parent.get_hierarchical_name()
                if parent_name is None:
                    return self.name
                else:
                    return parent_name + " -> " + self.name

        def get_level(self):
            if self.parent is None:
                return 1
            else:
                parent_name = self.parent.get_level()
                if parent_name is None:
                    return 1
                else:
                    return parent_name + 1

        def get_parent_id(self):
            return self.parent_id

        def has_parent(self):
            return self.parent_id is not None

    def write_table(self, services, date, product_names, yt):
        schema = [
            {"name": "date", "type": "string"},
            {"name": "abc_id", "type": "int64"},
            {"name": "abc_name", "type": "string"},
            {"name": "abc_level_1", "type": "string"},
            {"name": "abc_level_2", "type": "string"},
            {"name": "abc_level_3", "type": "string"},
            {"name": "abc_level_4", "type": "string"},
            {"name": "abc_level_5", "type": "string"},
            {"name": "abc_level_6", "type": "string"},
            {"name": "product", "type": "string"},
            {"name": "ordertype", "type": "int16"},
            {"name": "revenue", "type": "double"},
            {"name": "revenue_with_self", "type": "double"},
            {"name": "self_revenue", "type": "double"},
            {"name": "abc_parent", "type": "int64"},
            {"name": "is_oebs", "type": "boolean"},
            {"name": "is_vs", "type": "boolean"},
            {"name": "vs", "type": "int64"}

        ]

        rows = []

        for id, service in services.items():
            hn = service.get_hierarchical_name().split(" -> ")
            last_name = []
            lns = None
            if len(hn) > 5:
                for i in range(5, len(hn), 1):
                    last_name.append(hn[i])
                lns = " -> ".join(last_name)

            for slice in product_names:
                rows.append({
                    "date": date,
                    "abc_id": service.id,
                    "abc_name": service.name,
                    "abc_level_1": hn[0] if len(hn) > 0 else None,
                    "abc_level_2": hn[1] if len(hn) > 1 else None,
                    "abc_level_3": hn[2] if len(hn) > 2 else None,
                    "abc_level_4": hn[3] if len(hn) > 3 else None,
                    "abc_level_5": hn[4] if len(hn) > 4 else None,
                    "abc_level_6": lns,
                    "abc_parent": service.parent_id,
                    "ordertype": slice.order_type,
                    "product": slice.prouct,
                    "self_revenue": service.get_revenue(by_slice=slice),
                    "revenue_with_self": service.calc_child(with_self=True, by_slice=slice),
                    "revenue": service.calc_child(with_self=False, by_slice=slice),
                    "is_oebs": service.is_oebs,
                    "is_vs": service.is_vs,
                    "vs": service.vs,
                })

        table = "{}{}".format(self.Parameters.dl_table_prefix, date)
        if yt.exists(table):
            yt.remove(table)
        yt.create("table", table, attributes={"schema": schema})
        yt.write_table(yt.TablePath(table, append=True), rows)

    def calc_dwh(self, date, yt):
        services = {}
        vs = [675, 847, 850, 863, 909, 1455, 1920, 2608, 2825, 4446, 4447, 5782, 31735, 31736, 31752, 31797, 32066, 32607, 32924, 32986, 36349]

        abc_all_table = list(yt.read_table('//home/statkey/moneymap/abc'))

        revenue_all_table = list(yt.read_table('{}{}'.format(self.Parameters.revenue_table_prefix, date)))

        services[842] = self.AbcEntity(842, "Песочница", None)

        for abc in abc_all_table:
            services[abc["abc_id"]] = self.AbcEntity(abc["abc_id"], abc["ru_name"], abc["parent"], is_oebs=abc["synced_with_oebs"])

        skip_ids = [2289, 3366]

        for id, service in services.items():
            if id in skip_ids:
                continue
            if service.has_parent():
                services[service.get_parent_id()].add_child(service)

        product_names = set()

        for revenue in revenue_all_table:
            if revenue["revenue_gross_sum"] is not None and revenue["abc_oebs_id"] is not None:
                if revenue["abc_oebs_id"] == -1:
                    continue
                slice = self.Slice(revenue["ordertype"], revenue["product_name"])
                product_names.add(slice)
                services[revenue["abc_oebs_id"]].set_revenue(slice, revenue["revenue_gross_sum"])

        for service in services.items():
            if service[1].id in vs:
                service[1].is_vs = True
                service[1].setVsForChild(service[1].id)

        print("")

        self.write_table(services, date, product_names, yt)

    def run_yql_query(self, query, server, token, title):
        from yql.api.v1.client import YqlClient
        yql_client = YqlClient(db=server, token=token)
        logging.info('Make query:\n%s', query)
        request = yql_client.query(query, syntax_version=1, title=title)
        request.encoding = 'utf-8'
        request.run()

        if not request.get_results().is_success:
            error_description = '\n'.join([str(err) for err in request.get_results().errors])
            logging.error(error_description)
            raise RuntimeError(error_description)

        return request.get_results().table

    def prepare_dl(self, token):
        query = """
        use {};

        $dwh_table_prefix = "{}";
        $table_names = {};

        DEFINE ACTION $prepare_dl($table) AS
            $dwh_table = $dwh_table_prefix || "/" || $table;
            $dash_table = "{}" || $table;

            INSERT INTO $dash_table WITH TRUNCATE
            SELECT
                dwh.product_name as product_name,
                dwh.product as product,
                dwh.page_service_code as page_service_code,
                dwh.ordertype as ordertype,
                dwh.abc_oebs_id as abc_oebs_id,
                mm_abc.ru_name as abc_oebs_id_ru_name,
                sum(dwh.revenue_gross) as revenue_gross_sum
            from $dwh_table as dwh
            left join `home/statkey/moneymap/snapshots/last/abc` as mm_abc on dwh.abc_oebs_id = mm_abc.abc_id
            group by
                dwh.product_name,
                dwh.product,
                dwh.page_service_code,
                dwh.ordertype,
                dwh.abc_oebs_id,
                mm_abc.ru_name;
        END DEFINE;

        EVALUATE FOR $table IN $table_names DO BEGIN
            DO $prepare_dl($table)
        END DO;
        """.format(self.Parameters.yt_cluster, self.Parameters.dwh_table_prefix, "[\"{}\"]".format("\", \"".join(self.Parameters.dates)), self.Parameters.revenue_table_prefix)

        self.run_yql_query(query, self.Parameters.yt_cluster, token, "MM: Prepare DL | YQL")

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={"Name": "MoneyMapDatalens", "release": ctt.ReleaseStatus.STABLE},
        ).first()

    def on_execute(self):
        import yt.wrapper as yt

        token = sdk2.Vault.data(YQL_TOKEN_OWNER, YQL_TOKEN_NAME)
        yt.config.config['token'] = token
        yt.config.set_proxy(self.Parameters.yt_cluster)

        self.prepare_dl(token)

        for date in self.Parameters.dates:
            self.calc_dwh(date, yt)
