# -*- coding: utf-8 -*-
import logging

import requests
import sandbox.common.types.task as ctt
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import task_env
from sandbox.sandboxsdk import environments

MONEYMAP_URL = "http://moneymap.yandex-team.ru"

YQL_TOKEN_OWNER = 'STATKEY'
YQL_TOKEN_NAME = 'YQL_TOKEN'

MM_PROD_TVM_ID = 2024387
MM_DEV_TVM_ID = 2026174


class MoneyMapLoadFreshEntities(sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        cores = 1
        ram = 4096

        environments = [
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0')
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        yt_cluster = sdk2.parameters.String('Yt cluster', default='hahn')
        date_string_for_snapshot = sdk2.parameters.String('Date for snapshot YYYY-MM-DD', default=None)
        debug_options = sdk2.parameters.Bool(
            "Debug options",
            default_value=False,
        )

    def getServiceTicket(self):
        import tvmauth

        c = tvmauth.TvmClient(
            tvmauth.TvmApiClientSettings(
                self_tvm_id=MM_DEV_TVM_ID,
                enable_service_ticket_checking=True,
                enable_user_ticket_checking=tvmauth.BlackboxEnv.Test,
                self_secret=sdk2.yav.Secret('sec-01exz24p3nny9ssdtz0xhq6mw3').data()['client_secret'],
                dsts={"MM": MM_PROD_TVM_ID},
            )
        )
        return c.get_service_ticket_for("MM")

    def getRequestMM(self, api_path):
        retry_strategy = Retry(
            total=3,
            status_forcelist=[429, 500, 502, 503, 504],
            method_whitelist=["GET"],
            backoff_factor=2
        )

        service_ticket = self.getServiceTicket()

        adapter = HTTPAdapter(max_retries=retry_strategy)
        http = requests.Session()
        http.mount("https://", adapter)
        http.mount("http://", adapter)

        logging.info("Start request to {}".format(api_path))
        api_url = MONEYMAP_URL + api_path

        # get service ticket

        req = http.get(url=api_url, headers={'X-Ya-Service-Ticket': service_ticket})
        logging.info("Finish request to {}, with code {}".format(api_path, req.status_code))
        logging.info("Response is: {}".format(req.content))
        if (req.status_code == 200):
            return
        raise TaskFailure("Request to {} is fail".format(api_url))

    def loadFreshEntities(self):
        mm_url = "/api/v1/yt"
        fresh_entities = [
            "{url}/preload_pages?force=false&ytPath=//home/statkey/moneymap/new/{date}/pages_for_mark",
            "{url}/preload_blocks?force=false&ytPath=//home/statkey/moneymap/new/{date}/blocks_for_mark"
        ]
        for entity_api_template in fresh_entities:
            self.getRequestMM(entity_api_template.format(url=mm_url, date=self.Parameters.date_string_for_snapshot))

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={"Name": "MoneyMapLoadFreshEntities", "release": ctt.ReleaseStatus.STABLE},
        ).first()

    def on_execute(self):
        self.loadFreshEntities()
