# -*- coding: utf-8 -*-
import os
import requests
import json
import datetime
import calendar

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
import logging

from sandbox import sdk2
from sandbox.projects.common import task_env
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import environments
from sandbox.common.errors import TaskFailure

MONEYMAP_URL = "http://moneymap.in.yandex-team.ru"
MONEYMAP_DUMP_API = "/api/v1/yt/dump_"
MONEYMAP_UPDATE_ABC_API = "/api/v1/abc/fetch_services"
MONEYMAP_UPDATE_PAGES_FROM_PI_API = "/api/v1/yt/partner/preload_page_names"
MONEYMAP_UPDATE_BLOCKS_FROM_PI_API = "/api/v1/yt/partner/preload_block_names"
MONEYMAP_UPDATE_ZEN = "/api/v1/yt/zen/preload_entities"

ARTMON_ABC_OEBS = 'abc_oebs_id_artmon'
ARTMON_ABC = 'abc_id_artmon'
ARTMON_PLATFORM = 'platform_artmon'
ARTMON_OS = 'os_artmon'

ARTMON_ABC_OEBS_USED = 'abc_oebs_used'
ARTMON_ABC_USED = 'abc_used'
ARTMON_PLATFORM_USED = 'platform_used'
ARTMON_OS_USED = 'os_used'


YQL_TOKEN_OWNER = 'STATKEY'
YQL_TOKEN_NAME = 'YQL_TOKEN'

MODE_MAKE_SNAPSHOT = "make_snapshot"
MODE_COPY_TO_ARNOLD = "copy_to_arnold"

SOLOMON_FORMAT = "%Y-%m-%dT%H:%M:%SZ"

PAGES = "pages"
BLOCKS = "blocks"
ENTITY_TYPES = [PAGES, BLOCKS]

MM_PROD_TVM_ID = 2024387
MM_DEV_TVM_ID = 2026174


class MoneyMapPrepareAndMonitoringTable(sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        cores = 1
        ram = 4096

        environments = [
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0')
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.RadioGroup("History path date template") as mode:
            mode.values[MODE_MAKE_SNAPSHOT] = mode.Value(MODE_MAKE_SNAPSHOT, default=True)
            mode.values[MODE_COPY_TO_ARNOLD] = mode.Value(MODE_COPY_TO_ARNOLD)
        segmentation_version = sdk2.parameters.String('Version of segmentation', default='6.3.1')
        money_map_tables_path = sdk2.parameters.String('Path for moneymap tables', default='//home/statkey/moneymap')
        rsya_tables_path = sdk2.parameters.String('Path for RSYA tables',
                                                  default='//home/comdep-analytics/YAN/segmentation/static/v6.3.1')
        date_string_for_snapshot = sdk2.parameters.String('Date for snapshot YYYY-MM-DD', default=None)
        yt_cluster = sdk2.parameters.String('Yt cluster', default='hahn')
        with sdk2.parameters.Group("Solomon token") as solomon_token_block:
            solomon_token_secret_owner = sdk2.parameters.String("Owner of sb-vault-secret with solomon token",
                                                                required=True, default="USERSESSIONSTOOLS")
            solomon_token_secret_name = sdk2.parameters.String("Name of sb-vault-secret with solomon token",
                                                               required=True,
                                                               default="robot_make_sessions_solomon_token")
        debug_options = sdk2.parameters.Bool(
            "Debug options",
            default_value=False,
        )

    def getServiceTicket(self):
        import tvmauth

        c = tvmauth.TvmClient(
            tvmauth.TvmApiClientSettings(
                self_tvm_id=MM_DEV_TVM_ID,
                enable_service_ticket_checking=True,
                enable_user_ticket_checking=tvmauth.BlackboxEnv.Test,
                self_secret=sdk2.yav.Secret('sec-01exz24p3nny9ssdtz0xhq6mw3').data()['client_secret'],
                dsts={"MM": MM_PROD_TVM_ID},
            )
        )
        return c.get_service_ticket_for("MM")

    def getRequestMM(self, api_path):
        retry_strategy = Retry(
            total=3,
            status_forcelist=[429, 500, 502, 503, 504],
            method_whitelist=["GET"],
            backoff_factor=2
        )

        service_ticket = self.getServiceTicket()

        adapter = HTTPAdapter(max_retries=retry_strategy)
        http = requests.Session()
        http.mount("https://", adapter)
        http.mount("http://", adapter)

        logging.info("Start request to {}".format(api_path))
        api_url = MONEYMAP_URL + api_path

        # get service ticket

        req = http.get(url=api_url, headers={'X-Ya-Service-Ticket': service_ticket})
        logging.info("Finish request to {}, with code {}".format(api_path, req.status_code))
        logging.info("Response is: {}".format(req.content))
        if (req.status_code == 200):
            return
        raise TaskFailure("Request to {} is fail".format(api_url))

    def dumpDynTable(self, src, dst):
        import yt.wrapper as yt

        token = sdk2.Vault.data(YQL_TOKEN_OWNER, YQL_TOKEN_NAME)
        yt.config.config['token'] = token
        yt.config.set_proxy(self.Parameters.yt_cluster)

        if not yt.exists(dst):
            with yt.Transaction():
                yt.lock(src, mode="snapshot")
                ts = yt.get_attribute(src, "unflushed_timestamp") - 1
                yt.create("table", dst, recursive=True, attributes={
                    "optimize_for": yt.get_attribute(src, "optimize_for", default="lookup"),
                    "schema": yt.get_attribute(src, "schema"),
                    "_yt_dump_restore_pivot_keys": yt.get_attribute(src, "pivot_keys")
                })
                yt.run_merge(yt.TablePath(src, attributes={"timestamp": ts}), dst, mode="ordered")

    def getMMMonitoringTable(self, suffix, type_of_entity):
        return "//home/statkey/moneymap/monitorings/" + type_of_entity + "/" + suffix

    def getMMSnapshotTable(self, date, type_of_entity):
        return self.Parameters.money_map_tables_path + "/snapshots/{date}/{type_of_entity}".format(date=date,
                                                                                                   type_of_entity=type_of_entity)

    def getMMSnapshotDynamicTable(self, date, type_of_entity):
        return self.Parameters.money_map_tables_path + "/snapshots/{date}/dynamic/{type_of_entity}".format(date=date,
                                                                                                           type_of_entity=type_of_entity)

    def getMMSnapshotDir(self):
        return self.Parameters.money_map_tables_path + "/snapshots"

    def getMMTable(self, suffix):
        return self.Parameters.money_map_tables_path + "/" + suffix

    def createSnapshotFromDynTable(self, entity_type):
        src_table = self.getMMTable("{}".format(entity_type))
        dst_table = self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, entity_type)
        self.dumpDynTable(src_table, dst_table)

    def prepare_yql_query(self, type_of_entity):
        block_revenue = """
            $revenue_raw = (
                select
                    network_type,
                    ownership_type,
                    c.page_id           as page_id,
                    c.block_id          as block_id,
                    internal_service,
                    abc_oebs_id,
                    sum(c.revenue)      as revenue
                from `//home/comdep-analytics/YAN/cubes/yan/latest/cooked/v{segmentation_version}/cooked_yan_cube` as c
                left join `//home/comdep-analytics/YAN/segmentation/static/v{segmentation_version}/pages` as pp on pp.page_id = c.page_id
                left join `//home/comdep-analytics/YAN/segmentation/static/v{segmentation_version}/blocks` as p on p.page_id = c.page_id and p.block_id = c.block_id
                left join `{page_snapshot_table}` as mm on mm.page_id = c.page_id
                left join `{block_snapshot_table}` as m on m.page_id = c.page_id and m.block_id = c.block_id
                where c.report_date between $date_start and $date_end
                  and c.code_type != 'plain-awaps'
                group by
                    coalesce(p.network_type, pp.network_type)     as network_type,
                    coalesce(p.ownership_type, pp.ownership_type) as ownership_type,
                    c.page_id,
                    c.block_id,
                    if(p.is_vh_turbo, null, coalesce(p.internal_service, pp.internal_service)) as internal_service,
                    if(m.page_id is not null, m.abc_oebs_id, mm.abc_oebs_id) as abc_oebs_id
            );

            $revenue = (
                select
                    r.network_type      as network_type,
                    r.ownership_type    as ownership_type,
                    r.page_id           as page_id,
                    r.block_id          as block_id,
                    r.internal_service  as internal_service,
                    r.abc_oebs_id       as abc_oebs_id,
                    d.mm_name           as mm_name,
                    d.ba_name           as ba_name,
                    r.revenue           as revenue,
                from $revenue_raw as r
                left join `//home/statkey/moneymap/abc_mapping_dict/abc_servicename_dict_for_monitoring` as d on d.abc_id = r.abc_oebs_id
            );
        """.format(
            segmentation_version=self.Parameters.segmentation_version,
            page_snapshot_table=self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, PAGES),
            block_snapshot_table=self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, BLOCKS),
        )

        page_revenue = """
            $revenue = (
                select
                    p.network_type      as network_type,
                    p.ownership_type    as ownership_type,
                    c.page_id           as page_id,
                    p.internal_service  as internal_service,
                    m.abc_oebs_id       as abc_oebs_id,
                    d.mm_name           as mm_name,
                    d.ba_name           as ba_name,
                    sum(c.revenue)      as revenue
                from `//home/comdep-analytics/YAN/cubes/yan/latest/cooked/v{segmentation_version}/cooked_yan_cube` as c
                left join `//home/comdep-analytics/YAN/segmentation/static/v{segmentation_version}/{type_of_entity}` as p on p.page_id = c.page_id
                left join `{page_snapshot_table}` as m on m.page_id = c.page_id
                left join `//home/statkey/moneymap/abc_mapping_dict/abc_servicename_dict_for_monitoring` as d on d.abc_id = m.abc_oebs_id
                where c.report_date between $date_start and $date_end
                  and c.code_type != 'plain-awaps'
                group by
                    p.network_type,
                    p.ownership_type,
                    c.page_id,
                    p.internal_service,
                    m.abc_oebs_id,
                    d.mm_name,
                    d.ba_name
            );
        """.format(
            type_of_entity=type_of_entity,
            segmentation_version=self.Parameters.segmentation_version,
            page_snapshot_table=self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, PAGES),
        )

        func1 = """
            $compare_ba_mm_service = ($row) -> {
                return case
                    when (($row.abc_oebs_id in (50, 1021, 33714, 31752) and  $row.network_type = 'main-search') or nvl($row.internal_service, '') =  nvl($row.ba_name, '')) then true else false
                end;
            };

            $get_abc_id_comparison = ($row) -> {
                return case
                    when $row.network_type = 'main-search' then
                        case
                            when $row.abc_oebs_id is null  then 'main_search_no_abc_id'
                            when $row.abc_oebs_id  = -1    then 'main_search_fake_abc_id'
                            when $row.abc_oebs_id  = 32066 then 'main_search_unassigned_abc_32066'
                            when $row.abc_oebs_id != -1    then 'main_search_with_abc_id'
                            else 'main_search_unknown'
                        end
                   when $row.network_type = 'yan' and $row.ownership_type = 'internal' then
                        case
                            when $row.abc_oebs_id is null  then 'internal_services_no_abc_id'
                            when $row.abc_oebs_id  = -1    then 'internal_services_fake_abc_id'
                            when $row.abc_oebs_id  = 32066 then 'internal_services_unassigned_abc_32066'
                            when $row.abc_oebs_id != -1    then 'internal_services_with_abc_id'
                            else 'internal_services_unknown'
                        end
                   when $row.network_type = 'yan' and $row.ownership_type = 'external' then
                        case
                            when $row.abc_oebs_id is null  then 'external_no_abc_id'
                            when $row.abc_oebs_id  = 32066 then 'external_unassigned_abc_32066'
                            when $row.abc_oebs_id  = -1    then 'external_fake_abc_id'
                            when $row.abc_oebs_id != -1    then 'external_with_abc_id'
                            else 'external_unknown'
                        end
                   else
                        case
                            when $row.abc_oebs_id is null  then 'unknown_no_abc_id'
                            when $row.abc_oebs_id  = -1    then 'unknown_fake_abc_id'
                            when $row.abc_oebs_id  = 32066 then 'unknown_unassigned_abc_32066'
                            when $row.abc_oebs_id != -1    then 'unknown_with_abc_id'
                            else 'unknown_unknown'
                        end
                   end;
        };
        """

        func2 = """
        $is_good = ($row) -> {
            return if(
                (($row.is_equal and $row.comparison_type in $good_cases) or
                ($row.abc_id in (50, 1021, 33714, 31752) and $row.comparison_type = 'main_search_with_abc_id')), true, false);
        };
        """

        yql_query = """
            use hahn;
            PRAGMA AnsiInForEmptyOrNullableItemsCollections;

            $date_start = '2019-01-01';
            $date_end   = '{date}';

            $good_cases = AsList('internal_services_with_abc_id', 'external_no_abc_id',
                                'main_search_unassigned_abc_32066', 'main_search_with_abc_id');

            {func_1}

            {func_2}

            {revenue}

        $services_compare_table = (
            select
                mm_name,
                abc.ru_name as abc_ru_name,
                abc.abc_id as abc_id,
                internal_service,
                page_id,
                {blocks}
                $compare_ba_mm_service(TableRow()) as is_equal,
                $get_abc_id_comparison(TableRow()) as comparison_type,
                revenue,
            from $revenue as r
            left join `//home/statkey/moneymap/abc` as abc on abc.abc_id = r.abc_oebs_id
        );


        INSERT INTO `{monitoring_table}` WITH TRUNCATE
        select
            mm_name,
            internal_service,
            is_equal,
            comparison_type,
            abc_ru_name,
            sum(revenue)   as revenue,
            count(page_id) as page_count
        from $services_compare_table
        group by
            is_equal,
            mm_name,
            internal_service,
            comparison_type,
            abc_ru_name
        order by
            is_equal,
            revenue,
            comparison_type,
            internal_service;

        INSERT INTO `{monitoring_table}_total` WITH TRUNCATE
        select
            is_good,
            sum(revenue)   as revenue,
            count(page_id) as block_count
        from $services_compare_table
        group by
            $is_good(TableRow()) as is_good;


        INSERT INTO `{monitoring_table}_by_{type_of_entity}` WITH TRUNCATE
        select
            *
        from $services_compare_table
        order by
            page_id;
        """.format(
            date=self.Parameters.date_string_for_snapshot,
            type_of_entity=type_of_entity,
            revenue=page_revenue if type_of_entity == PAGES else block_revenue,
            monitoring_table=self.getMMMonitoringTable(self.Parameters.date_string_for_snapshot, type_of_entity),
            func_1=func1,
            func_2=func2,
            blocks="block_id," if type_of_entity == BLOCKS else ""
        )

        return yql_query

    def run_yql_query(self, query, server, token, title):
        from yql.api.v1.client import YqlClient
        yql_client = YqlClient(db=server, token=token)
        logging.info('Make query:\n%s', query)
        request = yql_client.query(query, syntax_version=1, title=title)
        request.encoding = 'utf-8'
        request.run()

        if not request.get_results().is_success:
            error_description = '\n'.join([str(err) for err in request.get_results().errors])
            logging.error(error_description)
            raise RuntimeError(error_description)

        return request.get_results().table

    def compareMMAndRSYA(self, type_of_entity):
        token = sdk2.Vault.data(YQL_TOKEN_OWNER, YQL_TOKEN_NAME)
        os.environ["YT_TOKEN"] = token
        query = self.prepare_yql_query(type_of_entity)
        return self.run_yql_query(query, self.Parameters.yt_cluster, token,
                                  "MM: {type}. Compare with rsya | YQL".format(type=type_of_entity))

    def yt_read_table(self, token, cluster, table):
        from yt.wrapper import YtClient, JsonFormat
        client = YtClient(cluster, token)
        return client.read_table(table, format=JsonFormat(attributes={"encode_utf8": False}), raw=False)

    def dt2ts(self, dt):
        return calendar.timegm(dt.utctimetuple())

    def fill_solomon_element(self, solomon_arr, ts, type_of_entity, type_of_graph, abc, value):
        solomon_arr.append(
            {"labels": {"type_of_entity": type_of_entity, "type_of_graph": type_of_graph, "abc": abc}, "ts": ts,
             "value": value})

    def fill_solomon_element_new(self, solomon_arr, ts, type_of_entity, type_of_graph, is_equal, comparison_type,
                                 internal_service, abc_ru_name, value):
        solomon_arr.append({"labels": {
            "type_of_entity": type_of_entity,
            "type_of_graph": type_of_graph,
            "is_equal": is_equal,
            "comparison_type": comparison_type,
            "internal_service": internal_service,
            "abc_name": abc_ru_name
        }, "ts": ts, "value": value})

    def GetSolomonToken(self):
        secret_owner = self.Parameters.solomon_token_secret_owner
        secret_name = self.Parameters.solomon_token_secret_name
        token = sdk2.Vault.data(secret_owner, secret_name)
        return token

    def send_to_solomon(self, sensors):
        token = self.GetSolomonToken()
        SOLOMON_URL = 'http://solomon.yandex.net/api/v2/push?project=statkey&cluster=main&service=moneymap'
        solomon_json = {
            "commonLabels": {
                "time_period": "1d"
            },
            "sensors": []
        }
        solomon_json["sensors"] = sensors

        logging.info("Sending data to solomon")
        headers = {
            "Content-type": "application/json",
            "Authorization": "OAuth {}".format(token)
        }
        logging.info("Data to solomon: " + json.dumps(solomon_json, ensure_ascii=False))
        response = requests.post(SOLOMON_URL, json=solomon_json, headers=headers)
        logging.info(response.content)
        response.raise_for_status()
        logging.info("Sent all to solomon")

    def sendToSolomonNEW(self, type_of_entity):
        solomon_arr = []
        token = sdk2.Vault.data(YQL_TOKEN_OWNER, YQL_TOKEN_NAME)
        comparison_arr = self.yt_read_table(token, self.Parameters.yt_cluster,
                                            self.getMMMonitoringTable(self.Parameters.date_string_for_snapshot,
                                                                      type_of_entity))
        ts = self.dt2ts(datetime.datetime.strptime(self.Parameters.date_string_for_snapshot, "%Y-%m-%d"))
        for result in comparison_arr:
            is_equal = result["is_equal"]
            comparison_type = result["comparison_type"]
            internal_service = result["internal_service"] if result["internal_service"] else "NULL"
            abc_ru_name = result["abc_ru_name"] if result["abc_ru_name"] else "NULL"
            logging.info("abc_ru_name: " + abc_ru_name)
            revenue = result["revenue"]
            count = result["page_count"]
            self.fill_solomon_element_new(
                solomon_arr, ts, type_of_entity,
                "revenue", is_equal, comparison_type,
                internal_service, abc_ru_name, revenue)
            self.fill_solomon_element_new(
                solomon_arr, ts, type_of_entity,
                "count", is_equal, comparison_type,
                internal_service, abc_ru_name, count)

        total_yt = self.yt_read_table(token, self.Parameters.yt_cluster,
                                      self.getMMMonitoringTable(self.Parameters.date_string_for_snapshot + "_total",
                                                                type_of_entity))
        total_revenue = 0
        total_count = float(0)
        total_arr = []
        for result in total_yt:
            total_arr.append(result)
            is_good = result["is_good"]
            revenue = result["revenue"]
            count = result["block_count"]
            total_count += count
            total_revenue += revenue
            solomon_arr.append({"labels": {
                "type_of_entity": type_of_entity,
                "type_of_graph": "total_revenue",
                "is_good": is_good,
            }, "ts": ts, "value": revenue})
            solomon_arr.append({"labels": {
                "type_of_entity": type_of_entity,
                "type_of_graph": "total_count",
                "is_good": is_good,
            }, "ts": ts, "value": count})

        for result in total_arr:
            is_good = result["is_good"]
            revenue = (result["revenue"] / total_revenue) * 100
            count = (result["block_count"] / total_count) * 100
            solomon_arr.append({"labels": {
                "type_of_entity": type_of_entity,
                "type_of_graph": "total_percent_of_revenue",
                "is_good": is_good,
            }, "ts": ts, "value": revenue})
            solomon_arr.append({"labels": {
                "type_of_entity": type_of_entity,
                "type_of_graph": "total_percent_of_count",
                "is_good": is_good,
            }, "ts": ts, "value": count})

        self.send_to_solomon(solomon_arr)

    def create_link_for_last_table(self, cluster):
        import yt.wrapper as yt

        token = sdk2.Vault.data(YQL_TOKEN_OWNER, YQL_TOKEN_NAME)
        yt.config.config['token'] = token
        yt.config.set_proxy(cluster)

        link_path = self.getMMSnapshotDir() + "/last"

        if yt.exists(link_path):
            yt.remove(link_path)
        dates = yt.list(self.getMMSnapshotDir())
        last_date = max(dates)
        last_date_path = self.getMMSnapshotDir() + "/" + last_date

        yt.link(last_date_path, link_path)

    def copy_to_arnold(self):
        from yt.transfer_manager.client import TransferManager

        client = TransferManager(token=sdk2.Vault.data(YQL_TOKEN_OWNER, YQL_TOKEN_NAME))

        self.Context.tm_task_ids = []

        entities = ENTITY_TYPES + ["abc"]

        for entity in entities:
            table = self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, entity)
            task_id = client.add_task(
                source_cluster=self.Parameters.yt_cluster,
                source_table=table,

                destination_cluster="arnold",
                destination_table=table,

                sync=True,
                params={
                    'queue_name': 'statbox-copy',
                    'copy_spec': {
                        'pool': 'statbox-cooked-logs-batch',
                    },
                    'schema_inference_mode': 'from_input',
                },
            )
            self.Context.tm_task_ids.append(task_id)

        self.create_link_for_last_table("arnold")

    def create_dynamic_tables_for_cluster(self, cluster):
        date = self.Parameters.date_string_for_snapshot
        for entity in ENTITY_TYPES:
            self.create_dynamic_tables(
                cluster,
                self.getMMSnapshotTable(date, entity),
                self.getMMSnapshotDynamicTable(date, entity)
            )

    def create_dynamic_tables(self, cluster, src, dst):
        import yt.wrapper as yt

        token = sdk2.Vault.data(YQL_TOKEN_OWNER, YQL_TOKEN_NAME)
        yt.config.config['token'] = token
        yt.config.set_proxy(cluster)

        schema = yt.get_attribute(src, "schema")
        schema.attributes["unique_keys"] = True

        # Create destination table.
        if not yt.exists(dst):
            yt.create("table", dst, recursive=True, attributes={
                "optimize_for": yt.get_attribute(src, "optimize_for", default="lookup"),
                "schema": schema
            })
        else:
            yt.remove(dst)
            yt.create("table", dst, recursive=True, attributes={
                "optimize_for": yt.get_attribute(src, "optimize_for", default="lookup"),
                "schema": schema
            })
        # Make blocks smaller (required for real-time lookups).
        yt.run_merge(src, dst, mode="ordered", spec={
            "job_io": {"table_writer": {"block_size": 256 * 2 ** 10, "desired_chunk_size": 100 * 2 ** 20}},
            "force_transform": True,
            "schema_inference_mode": "from_output"
        })
        # Make table dynamic.
        yt.alter_table(dst, dynamic=True)

        yt.mount_table(dst, sync=True)
        yt.set(dst + "/@forced_compaction_revision", 1)
        yt.remount_table(dst)

    def prepare_abc_for_ui_query(self):
        return """
            $abcs = (
                SELECT abc_id from `home/statkey/moneymap/snapshots/last/blocks`
                GROUP BY abc_id
                UNION ALL
                SELECT abc_id from `home/statkey/moneymap/snapshots/last/pages`
                GROUP BY abc_id
            );

            $oebs = (
                SELECT abc_oebs_id from `home/statkey/moneymap/snapshots/last/blocks`
                GROUP BY abc_oebs_id
                UNION ALL
                SELECT abc_oebs_id from `home/statkey/moneymap/snapshots/last/pages`
                GROUP BY abc_oebs_id
            );

            $platform = (
                SELECT platform from `home/statkey/moneymap/snapshots/last/blocks`
                GROUP BY platform
                UNION ALL
                SELECT platform from `home/statkey/moneymap/snapshots/last/pages`
                GROUP BY platform
            );

            $os = (
                SELECT os from `home/statkey/moneymap/snapshots/last/blocks`
                GROUP BY os
                UNION ALL
                SELECT os from `home/statkey/moneymap/snapshots/last/pages`
                GROUP BY os
            );

            INSERT INTO `{snapshot_abc_table}` WITH TRUNCATE
            SELECT
                *
            from `{abc_snapshot_table}` as abc
            INNER JOIN (
                select abc_id from $abcs GROUP BY abc_id
            ) as used_abc using(abc_id)
            order by abc_id;

            INSERT INTO `{snapshot_abc_oebs_table}` WITH TRUNCATE
            SELECT
                *
            from `{abc_snapshot_table}` as abc
            INNER JOIN (
                select abc_oebs_id from $oebs GROUP BY abc_oebs_id
            ) as used_abc on used_abc.abc_oebs_id = abc.abc_id
            order by abc_id;

            INSERT INTO `{snapshot_platform_table}` WITH TRUNCATE
            select platform, "-" as cap from $platform
            where platform is not null
            GROUP BY platform
            order by platform;

            INSERT INTO `{snapshot_os_table}` WITH TRUNCATE
            select os, "-" as cap from $os
            where os is not null
            GROUP BY os
            order by os;
        """.format(
            abc_snapshot_table=self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, "abc"),
            snapshot_abc_oebs_table=self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, ARTMON_ABC_OEBS),
            snapshot_abc_table=self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, ARTMON_ABC),
            snapshot_platform_table=self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, ARTMON_PLATFORM),
            snapshot_os_table=self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, ARTMON_OS),
        )

    def prepare_abc_for_artmon_ui(self, cluster):
        token = sdk2.Vault.data(YQL_TOKEN_OWNER, YQL_TOKEN_NAME)
        os.environ["YT_TOKEN"] = token
        self.run_yql_query(self.prepare_abc_for_ui_query(), cluster, token, "MM: Prepare artmon | YQL")
        table_names = {
            ARTMON_ABC: ARTMON_ABC_USED,
            ARTMON_ABC_OEBS: ARTMON_ABC_OEBS_USED,
            ARTMON_OS: ARTMON_OS_USED,
            ARTMON_PLATFORM: ARTMON_PLATFORM_USED
        }
        for table, used_table in table_names.iteritems():
            self.create_dynamic_tables(
                cluster,
                self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, table),
                self.getMMSnapshotDynamicTable(self.Parameters.date_string_for_snapshot, used_table)
            )
        # self.create_dynamic_tables(
        #     cluster,
        #     self.getMMSnapshotTable(self.Parameters.date_string_for_snapshot, ARTMON_ABC),
        #     self.getMMSnapshotDynamicTable(self.Parameters.date_string_for_snapshot, "abc_used")
        # )

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={"Name": "MoneyMapPrepareAndMonitoringTable", "release": ctt.ReleaseStatus.STABLE},
        ).first()

    def on_execute(self):
        if self.Parameters.mode == MODE_MAKE_SNAPSHOT:
            self.getRequestMM(MONEYMAP_UPDATE_ABC_API)
            self.getRequestMM(MONEYMAP_UPDATE_PAGES_FROM_PI_API)
            self.getRequestMM(MONEYMAP_UPDATE_BLOCKS_FROM_PI_API)
            self.getRequestMM(MONEYMAP_UPDATE_ZEN)
            self.createSnapshotFromDynTable("abc")
            for entity_type in ENTITY_TYPES:
                self.createSnapshotFromDynTable(entity_type)
                # self.compareMMAndRSYA(entity_type)
                # self.sendToSolomonNEW(entity_type)
        if self.Parameters.mode == MODE_COPY_TO_ARNOLD:
            self.create_link_for_last_table(self.Parameters.yt_cluster)
            self.copy_to_arnold()
            self.create_dynamic_tables_for_cluster("hahn")
            self.prepare_abc_for_artmon_ui("hahn")
            self.create_dynamic_tables_for_cluster("arnold")
            self.prepare_abc_for_artmon_ui("arnold")
