from functools import cmp_to_key


class HTMLFromYTGenerator:
    """Generator of html tables from YT table."""

    class Row:
        def __init__(self, row_dict, attributes=None):
            """
            :param row_dict: dict<row_key, string or `raw(<some html code>)'>.
                             ATTENTION: If you want more difficult value then string (link or image, for example)
                                        you can give value="raw('<some html code>')". "raw" is function from
                                        'dominate.util' which transform string to part of html code. Example of
                                        "raw" usage:
                                        value = raw('<a href={href}>Example</a>'.format(href=link))
            :param attributes: dict<row_attribute_key, row_attribute_value>.
            """
            if attributes is None:
                attributes = {}
            self.dict = row_dict
            self.attributes = attributes

    def __init__(self, cluster, src_yt_table_path, yt_token=None):
        from yt.wrapper import YtClient
        """
        Constructor, which saves list of dicts-rows, all keys and values are strings.

        :param cluster: string with cluster name.
        :param src_yt_table_path: string with path to source YT table.
        """
        client = YtClient(proxy=cluster, config={"tabular_data_format": "yson"}, token=yt_token)
        self.rows = list(client.read_table(client.TablePath(src_yt_table_path)))

    def generate(self, keys_list, new_keys=None, src_rows_filter=None, res_rows_comparator=None,
                 coloring_equal_rows=False):
        """
        Generate html table.
        Assumption: source YT table isn't empty

        :param keys_list: list of keys which contains all keys of result table and sets order of keys.
                          If key wasn't found in YT table or in new_keys param, it's value is "null"
        :param new_keys: dictionary of key names and the function which to get value using YT table row.
                         ATTENTION: value here is string or `raw(<html code>)`.
                         ATTENTION: If you want more difficult value then string (link or image, for example)
                                    you can give value="raw('<some html code>')". "raw" is function from
                                    'dominate.util' which transform string to part of html code. Example of
                                    "raw" usage:
                                    value = raw('<a href={href}>Example</a>'.format(href=link))
                         If new_keys contains key from keys_to_move, it is ignored in keys_to_move
                         There isn't any new key if param is None.
        :param src_rows_filter: function which filters rows we want to move to html table
        :param res_rows_comparator: comparator for sorting of result rows.
                                    Rows order is random if param is None.
        :param coloring_equal_rows: flag to color equal rows (equality is defined by res_rows_comparator) in one color
                                    if param is true and res_rows_comparator isn't None, otherwise not color.
        :return: string with html code.
        """
        filtered_rows = [row for row in self.rows if src_rows_filter is None or src_rows_filter(row)]

        sorted_filtered_rows = sorted(filtered_rows, key=cmp_to_key(res_rows_comparator))

        res_row_list = []
        color_flag = True  # True -- one color, False -- another color
        prev_row = None
        for row in sorted_filtered_rows:
            row_dict = {}
            for (new_key, get_key) in new_keys.items():
                row_dict[new_key] = get_key(row)
            for key in [k for k in keys_list if k not in new_keys.keys()]:
                row_dict[key] = str(row.get(key))

            if prev_row is not None and not (res_rows_comparator(row, prev_row) and res_rows_comparator(prev_row, row)):
                color_flag = not color_flag
            res_row_list.append(self.Row(
                row_dict,
                {"bgcolor": "CCCCFF" if color_flag else "99FFCC"} if coloring_equal_rows else None)
            )
            prev_row = row

        return HTMLFromYTGenerator.make_table_from_rows_list(keys_list, res_row_list)

    @staticmethod
    def make_table_from_rows_list(keys_list, rows_list, table_style_css_path=None):
        from dominate.tags import html, body, head, table, thead, th, tbody, tr, td, link

        """
        Make html table from HeadRow with keys and list of Rows.

        :param keys_list: list of keys which contains all keys of result table and sets order of keys.
                          Raises ValueError if set(keys of head_row) != every row key_set.
        :param rows_list: list of Rows.
        :param table_style_css_path: preparation for adding <style>. Now table_style must be None, otherwise -- ValueError.
        :return: string with html code.
        """

        for row in rows_list:
            if set(keys_list) != set(row.dict.keys()):
                raise ValueError("Row keys set '{row_keys_set}' doesn't match head row keys set '{keys_set}'"
                                 .format(row_keys_set=set(row.dict.keys()), keys_set=set(keys_list)))

        html_table = html()

        with html_table.add(head()):
            link(href="https://cdn.jsdelivr.net/npm/bootstrap@5.0.2/dist/css/bootstrap.min.css",
                 rel="stylesheet",
                 integrity="sha384-EVSTQN3/azprG1Anm3QDgpJLIm9Nao0Yz1ztcQTwFspd3yD65VohhpuuCOmLASjC",
                 crossorigin="anonymous")

        if table_style_css_path is not None:
            raise ValueError("Unsupport table_style_css_path param")

        with html_table.add(body(cls="body")).add(table(cls="table table-bordered")):
            with thead():
                th_row = tr()
                for key in keys_list:
                    th_row.add(th(key))

            with tbody():
                for row in rows_list:
                    trow = tr()
                    trow.attributes = row.attributes
                    for key in keys_list:
                        trow.add(td(row.dict[key]))

        return str(html_table)
