# coding=utf-8

import logging
from datetime import timedelta, datetime
from sandbox import sdk2
from sandbox.sdk2 import yav

from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task, task_env
from sandbox.projects.statkey.money_map.monitorings.html_report import HTMLFromYTGenerator


class MoneymapReport(sdk2.Resource):
    """ Plain text data """

EXTERNAL_RSYA = 36259


class MoneymapMonitorings(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        disk_space = 10000
        ram = 4 * 1024

    class Parameters(sdk2.Task.Parameters):
        report_date = sdk2.parameters.String(
            'Date YYYY-MM-DD',
            required=True,
        )
        outputPrefix = sdk2.parameters.String(
            'Prefix of path to output tables',
            default_value='home/statkey/moneymap/new'
        )
        ext_params = binary_task.binary_release_parameters(stable=True)
        YQLSecretID = sdk2.parameters.String(
            'YQL token secret ID',
            required=True
        )
        YQLTokenKey = sdk2.parameters.String(
            'YQL token key',
            required=True
        )
        YTSecretID = sdk2.parameters.String(
            'YT token secret ID',
            required=True
        )
        YTTokenKey = sdk2.parameters.String(
            'YT token key',
            required=True
        )
        StartrekTokenID = sdk2.parameters.String(
            'Tracker token secret ID',
            default='sec-01fqh9fqrz0m3jab1rqmhcqtw9',
            required=True
        )
        TicketAssignee = sdk2.parameters.String(
            'Unmarked entities ticket assignee',
            default='bagiro44',
            required=True
        )

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "MONEYMAP_MONITORINGS", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def prepare_alice_query(self):
        return """
            USE hahn;

            PRAGMA OrderedColumns;
            PRAGMA AnsiInForEmptyOrNullableItemsCollections;

            $dict_bs_pages_with_app_id = (
                SELECT
                    PageID AS page_id,
                    if(OptionsApp AND Unicode::IsUtf(Name), Name ) AS app_id,
                FROM `//home/yabs/dict/Page`
            );

            $alice_bs_pages = (
                SELECT
                    a.page_id AS page_id
                FROM
                    $dict_bs_pages_with_app_id AS a
                    INNER JOIN `//home/comdep-analytics/YAN/segmentation/static/v6.3.3/inputs/internal_appids` AS b ON b.app_id == a.app_id
            );

            $sdk_pi_pages_with_app_id = (
                SELECT
                    b.context_page_id AS page_id,
                    a.store_id AS app_id
                FROM
                    `//home/partner/dict/mysql_dump/{report_date}/mobile_app` AS a
                    INNER JOIN
                        `//home/partner/dict/mysql_dump/{report_date}/mobile_app_settings` AS b
                        ON a.id == b.application_id
            );

            $ssp_pi_pages_with_app_id = (
                SELECT
                    id        AS page_id,
                    store_id  AS app_id
                FROM `//home/partner/dict/mysql_dump/{report_date}/internal_mobile_app`
                WHERE id > 0
            );

            $pi_pages_with_app_id = (
                SELECT * FROM $sdk_pi_pages_with_app_id
                UNION ALL
                SELECT * FROM $ssp_pi_pages_with_app_id

            );

            $alice_pi_pages = (
                SELECT
                    page_id
                FROM
                    $pi_pages_with_app_id AS a
                    INNER JOIN `//home/comdep-analytics/YAN/segmentation/static/v6.3.3/inputs/internal_appids` AS b ON b.app_id == a.app_id
                GROUP BY CAST(a.page_id AS INT64) AS page_id
            );

            $alice_pages_not_in_MM = (
                SELECT
                    p.page_id AS page_id
                FROM ANY (
                    SELECT * FROM $alice_bs_pages
                    UNION ALL
                    SELECT * FROM $alice_pi_pages
                ) AS p LEFT JOIN `//home/statkey/moneymap/snapshots_tmp/{report_date}/pages` AS mm
                ON p.page_id == mm.page_id
                WHERE abc_oebs_id == -1 OR abc_oebs_id IS NULL or mm.abc_oebs_id == {external_rsya}
            );

            INSERT INTO `{outputPrefix}/{report_date}/not_marked_pages/alice`  WITH TRUNCATE
            SELECT
                page_id,
                "this is alice" as source
            FROM $alice_pages_not_in_MM
            where page_id not in [345519, 462015]
            order by page_id;
        """.format(
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix,
            external_rsya=EXTERNAL_RSYA,
        )

    def prepare_new_entities_query(self):
        return """
            USE hahn;

            PRAGMA OrderedColumns;

            $bs_pages_not_in_MM = (
                SELECT
                    p.PageID AS page_id
                FROM `//home/yabs/dict/Page` AS p
                LEFT ONLY JOIN `//home/statkey/moneymap/snapshots_tmp/{report_date}/pages` AS mm ON p.PageID == mm.page_id
            );

            $pi_pages_not_in_MM = (
                SELECT
                    p.id AS page_id
                FROM `//home/partner/dict/mysql_dump/{report_date}/all_pages` as p
                LEFT ONLY JOIN `//home/statkey/moneymap/snapshots_tmp/{report_date}/pages` AS mm ON p.page_id == mm.page_id
            );

            $pages_not_in_MM = (
                SELECT
                    nvl(CAST(p.page_id AS INT64), b.page_id) AS page_id
                FROM $pi_pages_not_in_MM AS p
                FULL JOIN $bs_pages_not_in_MM AS b ON p.page_id == b.page_id
            );

            INSERT INTO `{outputPrefix}/{report_date}/pages_for_mark` WITH TRUNCATE
            SELECT * FROM $pages_not_in_MM
            order by page_id;

            $bs_blocks_not_in_MM = (
                SELECT
                    p.PageID AS page_id,
                    p.ImpID AS block_id
                FROM `//home/yabs/dict/replica/PageImp` AS p
                LEFT ONLY JOIN `//home/statkey/moneymap/snapshots_tmp/{report_date}/blocks` AS mm
                ON p.PageID == mm.page_id AND p.ImpID == mm.block_id
            );

            $pi_blocks_not_in_MM = (
                SELECT
                    p.page_id AS page_id,
                    p.id AS block_id
                FROM `//home/partner/dict/mysql_dump/{report_date}/all_blocks` as p
                LEFT ONLY JOIN `//home/statkey/moneymap/snapshots_tmp/{report_date}/blocks` AS mm
                ON p.page_id == mm.page_id AND p.id == mm.block_id
            );

            $blocks_not_in_MM = (
                SELECT
                    nvl(CAST(p.page_id AS INT64), b.page_id) AS page_id,
                    nvl(p.block_id, b.block_id) AS block_id
                FROM $pi_blocks_not_in_MM AS p
                FULL JOIN $bs_blocks_not_in_MM AS b ON p.page_id == b.page_id AND p.block_id == b.block_id
            );

            INSERT INTO `{outputPrefix}/{report_date}/blocks_for_mark` WITH TRUNCATE
            SELECT blocks.* FROM $blocks_not_in_MM as blocks
            LEFT JOIN `//home/partner/dict/mysql_dump/{report_date}/all_pages` as pages on pages.page_id = blocks.page_id
            where block_id != 0 and pages.page_id is not NULL
            order by page_id, block_id;
        """.format(
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix
        )

    def prepare_vh_query(self):
        is_internal_block = """
            $is_internal_block = ($video_category_id) -> {
                return case
                    when $video_category_id = 1023 then false
                    else true
                end;
            };
        """

        return """
            {is_internal_block}

            $vh_pages_all = (
                SELECT
                    video_an_site.id as page_id,
                    1558 as abc_oebs_id,
                    1558 as abc_id,
                    "videoresources" as traffic_type,
                    "video" as traffic_sub_type,
                    "video" as traffic_sub_sub_type,
                    true as is_internal,
                    false as is_sandbox,
                    mm.abc_oebs_id as tmp_mm_oebs_id
                from `//home/partner/dict/mysql_dump/{report_date}/video_an_site` as video_an_site
                left join `//home/statkey/moneymap/snapshots_tmp/{report_date}/pages` as mm on mm.page_id = video_an_site.id
                where video_an_site.partner_type = "yahosting"
            );

            $all_vh_blocks = (
                SELECT
                    vh_pages.page_id as page_id,
                    pi_blocks.id as block_id,
                    NVL(vh_categories.abc_id, vh_pages.abc_oebs_id) as abc_oebs_id,
                    NVL(vh_categories.abc_id, vh_pages.abc_id) as abc_id,
                    "videoresources" as traffic_type,
                    "video" as traffic_sub_type,
                    "instream" as traffic_sub_sub_type,
                    pi_blocks.category_id as category_id,
                    $is_internal_block(pi_blocks.category_id) as is_internal,
                    if (NVL(vh_categories.abc_id, vh_pages.abc_oebs_id) = vh_pages.abc_oebs_id, true, false) as synced_with_page
                from $vh_pages_all as vh_pages
                left join `home/partner/dict/mysql_dump/{report_date}/all_blocks` as pi_blocks on pi_blocks.page_id = vh_pages.page_id
                left join `//home/statkey/moneymap/dict/vh_category_id_abc_id` as vh_categories on pi_blocks.category_id = vh_categories.id
                left join `//home/statkey/moneymap/snapshots_tmp/{report_date}/blocks` as mm on mm.page_id = pi_blocks.page_id and pi_blocks.id = mm.block_id
                where pi_blocks.id is not null and ((
                    mm.abc_oebs_id is null or mm.abc_oebs_id = -1 or mm.abc_oebs_id == 36259) and
                    (
                        NVL(vh_categories.abc_id, vh_pages.abc_oebs_id) != mm.abc_oebs_id or
                        NVL(vh_categories.abc_id, vh_pages.abc_id) != mm.abc_id)
                    )
                order by page_id, block_id
            );

            INSERT INTO `{outputPrefix}/{report_date}/not_marked_blocks/vh` WITH TRUNCATE
            SELECT
                page_id,
                block_id,
                "VH" as source,
                if(is_internal, abc_oebs_id, {external_rsya}) as abc_oebs_id,
                if(is_internal, abc_id, {external_rsya}) as abc_id,
                if(is_internal, traffic_type, null) as traffic_type,
                if(is_internal, traffic_sub_type, null) as traffic_sub_type,
                if(is_internal, traffic_sub_sub_type, null) as traffic_sub_sub_type,
                if(is_internal, category_id, null) as category_id,
                is_internal,
                false as is_sandbox,
                if(is_internal, synced_with_page, null) as synced_with_page
            from $all_vh_blocks
            order by page_id, block_id;

            INSERT INTO `{outputPrefix}/{report_date}/not_marked_pages/vh` WITH TRUNCATE
            SELECT
                vh_pages_all.*,
                "VH" as source
            from $vh_pages_all as vh_pages_all
            where vh_pages_all.tmp_mm_oebs_id is null or vh_pages_all.tmp_mm_oebs_id = -1 or
            vh_pages_all.tmp_mm_oebs_id == {external_rsya}
            order by vh_pages_all.page_id;
        """.format(
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix,
            is_internal_block=is_internal_block,
            external_rsya=EXTERNAL_RSYA,
        )

    def prepare_not_marked_mm_pages(self):
        return """
            USE hahn;

            PRAGMA OrderedColumns;

            INSERT INTO `{outputPrefix}/{report_date}/not_marked_pages/by_login` WITH TRUNCATE
            SELECT
                p.id AS page_id,
                "get from inner LOGINS" as source
            FROM `//home/partner/dict/mysql_dump/{report_date}/all_pages` AS p
            INNER JOIN `//home/partner/dict/inner_login` AS l ON p.login == l.login
            LEFT JOIN `//home/statkey/moneymap/snapshots_tmp/{report_date}/pages` AS mm ON p.id == mm.page_id
            where mm.is_internal = false and p.id not in [644525]
            order by page_id;
        """.format(
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix)

    def prepare_not_marked_mm_blocks(self):
        return """
            USE hahn;

            PRAGMA OrderedColumns;

            $pages_with_inner_login = (
                SELECT
                    p.id AS page_id,
                    "get from inner LOGINS" as source
                FROM `//home/partner/dict/mysql_dump/{report_date}/all_pages` AS p
                INNER JOIN `//home/partner/dict/inner_login` AS l ON p.login == l.login
            );

            INSERT INTO `{outputPrefix}/{report_date}/not_marked_blocks/not_marked_mm_blocks` WITH TRUNCATE
            SELECT
                b.page_id AS page_id,
                b.id AS block_id
            FROM `//home/partner/dict/mysql_dump/{report_date}/all_blocks` AS b
            INNER JOIN $pages_with_inner_login as ip ON b.page_id == ip.page_id
            LEFT ONLY JOIN `//home/statkey/moneymap/snapshots_tmp/{report_date}/blocks` AS mm
            ON b.page_id == mm.page_id AND b.id == mm.block_id
            WHERE b.id != 0 and b.page_id not in [644525]
            order by page_id, block_id;
        """.format(
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix)

    def prepare_pages_report(self):
        return """
            USE hahn;

            PRAGMA OrderedColumns;
            PRAGMA autocommit;

            INSERT INTO @union_table
            SELECT
                "" AS source,
                CAST(NULL AS INT64) AS page_id,
                CAST(NULL AS INT32) AS abc_id,
                CAST(NULL AS String) AS traffic_type,
                CAST(NULL AS String) AS traffic_sub_type,
                CAST(NULL AS String) AS traffic_sub_sub_type,
                CAST(NULL AS String) AS os,
                CAST(NULL AS String) AS platform,
                CAST(NULL AS INT32) AS abc_oebs_id,
                CAST(NULL AS BOOL) AS is_internal
            LIMIT 0;

            DEFINE ACTION $update_union_table($table_path) AS
                INSERT INTO @union_table
                SELECT TableName($table_path) AS source, t.* FROM $table_path AS t;
            END DEFINE;

            $auto_marking_table_paths_list = (
                SELECT AGGREGATE_LIST(Path)
                FROM FOLDER(`{outputPrefix}/{report_date}/pages`)
                WHERE Type == "table"
            );

            EVALUATE FOR $table_path IN $auto_marking_table_paths_list
                DO $update_union_table($table_path);

            $not_marked_pages = (SELECT
                CAST (page_id AS INT64) AS page_id,
                TableName() as source,
                false AS is_auto_marked,
                CAST(NULL AS BOOL) AS has_conflicts,
                CAST(NULL AS INT32) AS abc_id,
                CAST(NULL AS String) AS traffic_type,
                CAST(NULL AS String) AS traffic_sub_type,
                CAST(NULL AS String) AS traffic_sub_sub_type,
                CAST(NULL AS String) AS os,
                CAST(NULL AS String) AS platform,
                CAST(NULL AS INT32) AS abc_oebs_id,
                CAST(NULL AS BOOL) AS is_internal
            FROM RANGE(`{outputPrefix}/{report_date}/not_marked_pages`));

            INSERT INTO `{outputPrefix}/{report_date}/report_pages` WITH TRUNCATE
            SELECT
                *
            FROM $not_marked_pages AS a
            LEFT ONLY JOIN @union_table AS b ON a.page_id == b.page_id
            ORDER BY page_id;

            INSERT INTO `{outputPrefix}/{report_date}/report_pages`
            SELECT
                ut.page_id AS page_id,
                ut.source AS source,
                true AS is_auto_marked,
                d.has_conflicts AS has_conflicts,
                ut.* WITHOUT ut.page_id, ut.source
            FROM @union_table as ut
            INNER JOIN (
                SELECT
                    page_id,
                    COUNT(DISTINCT(source, abc_id, traffic_type, traffic_sub_type, traffic_sub_type, os, platform,
                        abc_oebs_id, is_internal)) > 1 AS has_conflicts
                FROM @union_table
                GROUP BY page_id
            ) AS d ON ut.page_id == d.page_id
            ORDER BY NOT has_conflicts, page_id;
        """.format(
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix)

    def prepare_blocks_report(self):
        return """
            USE hahn;

            PRAGMA OrderedColumns;
            PRAGMA autocommit;

            INSERT INTO @union_table
            SELECT
                "" AS source,
                CAST(NULL AS INT64) AS page_id,
                CAST(NULL AS INT64) AS block_id,
                CAST(NULL AS INT32) AS abc_id,
                CAST(NULL AS String) AS traffic_type,
                CAST(NULL AS String) AS traffic_sub_type,
                CAST(NULL AS String) AS traffic_sub_sub_type,
                CAST(NULL AS String) AS os,
                CAST(NULL AS String) AS platform,
                CAST(NULL AS INT32) AS abc_oebs_id,
                CAST(NULL AS BOOL) AS is_internal
            LIMIT 0;

            DEFINE ACTION $update_union_table($table_path) AS
                INSERT INTO @union_table
                SELECT TableName($table_path) AS source, t.* FROM $table_path AS t;
            END DEFINE;

            $auto_marking_table_paths_list = (
                SELECT AGGREGATE_LIST(Path)
                FROM FOLDER(`{outputPrefix}/{report_date}/blocks`)
                WHERE Type == "table"
            );

            EVALUATE FOR $table_path IN $auto_marking_table_paths_list
                DO $update_union_table($table_path);

            $not_marked_pages = (SELECT
                CAST (page_id AS INT64) AS page_id,
                CAST (block_id AS INT64) AS block_id,
                TableName() as source,
                false AS is_auto_marked,
                CAST(NULL AS BOOL) AS has_conflicts,
                CAST(NULL AS INT32) AS abc_id,
                CAST(NULL AS String) AS traffic_type,
                CAST(NULL AS String) AS traffic_sub_type,
                CAST(NULL AS String) AS traffic_sub_sub_type,
                CAST(NULL AS String) AS os,
                CAST(NULL AS String) AS platform,
                CAST(NULL AS INT32) AS abc_oebs_id,
                CAST(NULL AS BOOL) AS is_internal
            FROM RANGE(`{outputPrefix}/{report_date}/not_marked_blocks`));

            INSERT INTO `{outputPrefix}/{report_date}/report_blocks` WITH TRUNCATE
            SELECT
                CAST (page_id AS INT64) AS page_id,
                CAST (a.block_id AS INT64) AS block_id,
                "inner_blocks_not_marked" AS source,
                false AS is_auto_marked,
                CAST(NULL AS BOOL) AS has_conflicts,
                CAST(NULL AS INT32) AS abc_id,
                CAST(NULL AS String) AS traffic_type,
                CAST(NULL AS String) AS traffic_sub_type,
                CAST(NULL AS String) AS traffic_sub_sub_type,
                CAST(NULL AS String) AS os,
                CAST(NULL AS String) AS platform,
                CAST(NULL AS INT32) AS abc_oebs_id,
                CAST(NULL AS BOOL) AS is_internal
            FROM `{outputPrefix}/{report_date}/not_marked_blocks/not_marked_mm_blocks` AS a
            LEFT ONLY JOIN @union_table AS b ON a.page_id == b.page_id AND a.block_id == b.block_id
            ORDER BY page_id;

            INSERT INTO `{outputPrefix}/{report_date}/report_blocks`
            SELECT
                ut.page_id AS page_id,
                ut.block_id AS block_id,
                ut.source AS source,
                true AS is_auto_marked,
                d.has_conflicts AS has_conflicts,
                ut.* WITHOUT ut.page_id, ut.source, ut.block_id
            FROM @union_table as ut
            INNER JOIN (
                SELECT
                    page_id,
                    block_id,
                    COUNT(DISTINCT (source, abc_id, traffic_type, traffic_sub_type, traffic_sub_type, os, platform,
                        abc_oebs_id, is_internal)) > 1 AS has_conflicts
                FROM @union_table
                GROUP BY (page_id, block_id)
            ) AS d ON ut.page_id == d.page_id AND ut.block_id == d.block_id
            ORDER BY NOT has_conflicts, page_id, block_id;
        """.format(
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix)

    def prepare_main_search(self):
        get_abc = """
            $get_service = ($row) -> {
                return case
                    when $row.`PageID` in [467415, 620060] then 2728
                    when $row.`OptionsMainSerp` = true or $row.`PageID` in [90, 238] then 1021
                    when FIND(Unwrap($row.`Labels`), "suggest") is not null then 50
                    when $row.`OptionsAllAdsPage` = true then 33714
                    when $row.`TargetType` in [0] then 31752
                    else null
                end;
            };
        """

        return """
            PRAGMA AnsiInForEmptyOrNullableItemsCollections;

            use hahn;

            {get_abc}

            $search = (
                SELECT
                    PageID as page_id,
                    31752 as abc_oebs_id,
                    $get_service(JoinTableRow()) as abc_id,
                from `//home/yabs/dict/Page`
            );

            INSERT INTO `{outputPrefix}/{report_date}/not_marked_pages/main_search` WITH TRUNCATE
            SELECT
                main_search.page_id     AS page_id,
                "this is main_search" as source,
                main_search.abc_oebs_id AS abc_oebs_id,
                main_search.abc_id      AS abc_id,
                "main-search"           AS traffic_type,
                "main-search"           AS traffic_sub_type,
                "main-search"           AS traffic_sub_sub_type,
                CAST(NULL AS String)    AS platform,
                CAST(NULL AS String)    AS os,
                true                    AS is_internal,
                false AS is_sandbox
            from $search as main_search
            left join `//home/statkey/moneymap/snapshots_tmp/{report_date}/pages` as mm using(page_id)
            where main_search.abc_id is not null and (mm.abc_oebs_id = -1 or mm.abc_oebs_id is null
                                                        or mm.abc_oebs_id = {external_rsya}
                                                        or mm.abc_id != main_search.abc_id)
            order by page_id;""".format(
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix,
            get_abc=get_abc,
            external_rsya=EXTERNAL_RSYA,
        )

    def prepare_zen_page_query(self):
        return """
            use hahn;

            INSERT INTO `{outputPrefix}/{report_date}/not_marked_pages/zen` WITH TRUNCATE
            SELECT
                zen.page_id as page_id,
                "ZEN config" as source,
                zen.ABC as abc_oebs_id,
                zen.ABC as abc_id,
                mm.traffic_type as traffic_type,
                mm.traffic_sub_type as traffic_sub_type,
                mm.traffic_sub_sub_type as traffic_sub_sub_type,
                mm.platform as platform,
                mm.os as os,
                true as is_internal,
                false AS is_sandbox
            FROM `//home/recommender/zen/export/money_config_cumulative` as zen
            left join `//home/statkey/moneymap/snapshots_tmp/{report_date}/pages` as mm on zen.page_id = mm.page_id
            left join `//home/partner/dict/mysql_dump/{report_date}/all_pages` AS p on zen.page_id = p.id
            where (zen.ABC != mm.abc_oebs_id or mm.abc_oebs_id is null or mm.abc_oebs_id == {external_rsya})
            AND p.login not in ["dzen-platform"];""".format(
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix,
            external_rsya=EXTERNAL_RSYA,
        )
        # TODO dzen-platform is hack, remove it after zen sold

    def not_marked_pages_by_domain(self):
        return """
            use hahn;

            INSERT INTO `{outputPrefix}/{report_date}/not_marked_pages/by_domain` WITH TRUNCATE
            SELECT
                pi.page_id as page_id,
                "get from inner domains" as source
            from `home/partner/dict/mysql_dump/{report_date}/all_pages` as pi
            inner join `home/comdep-analytics/YAN/segmentation/static/v6.3.3/inputs/internal_domains` AS domains ON domains.domain = pi.domain
            left join `home/statkey/moneymap/pages` as mm on mm.page_id = pi.page_id
            WHERE mm.abc_oebs_id == -1 or mm.abc_oebs_id is null or mm.abc_oebs_id == {external_rsya};
        """.format(
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix,
            external_rsya=EXTERNAL_RSYA,
        )

    def diff_day_by_day(self):
        previous_day = (datetime.strptime(self.Parameters.report_date, '%Y-%m-%d') - timedelta(days=1)).strftime('%Y-%m-%d')
        eq_func = """$eq_fields = ($a, $b) -> {
            return case
                when $a != $b then "EDIT"
                when $a is null and $b is not null then "ADD"
                when $a is not null and $b is null then "DELETE"
                when $a = $b then "NO_CHANGES"
                when $a is null and $b is null then "NO_CHANGES"
                else "UNKNOWN"
            end;
        };"""
        return """
            USE hahn;
            PRAGMA OrderedColumns;

            {eq_func}

            $join_pages = (
                SELECT
                    new.page_id                 as page_id,
                    old.abc_oebs_id             as old_abc_oebs_id,
                    new.abc_oebs_id             as new_abc_oebs_id,
                    old.abc_id                  as old_abc_id,
                    new.abc_id                  as new_abc_id,
                    old.traffic_type            as old_traffic_type,
                    new.traffic_type            as new_traffic_type,
                    old.traffic_sub_type        as old_traffic_sub_type,
                    new.traffic_sub_type        as new_traffic_sub_type,
                    old.traffic_sub_sub_type    as old_traffic_sub_sub_type,
                    new.traffic_sub_sub_type    as new_traffic_sub_sub_type,
                    old.platform                as old_platform,
                    new.platform                as new_platform,
                    old.os                      as old_os,
                    new.os                      as new_os,
                    old.is_internal             as old_is_internal,
                    new.is_internal             as new_is_internal,
                    $eq_fields(old.abc_oebs_id, new.abc_oebs_id) as is_eq_abc_oebs_id,
                    $eq_fields(old.abc_id, new.abc_id) as is_eq_abc_id,
                    $eq_fields(old.traffic_type, new.traffic_type) as is_eq_traffic_type,
                    $eq_fields(old.traffic_sub_type, new.traffic_sub_type) as is_eq_traffic_sub_type,
                    $eq_fields(old.traffic_sub_sub_type, new.traffic_sub_sub_type) as is_eq_traffic_sub_sub_type,
                    $eq_fields(old.os, new.os) as is_eq_os,
                    $eq_fields(old.is_internal, new.is_internal) as is_eq_is_internal
                from `//home/statkey/moneymap/snapshots_tmp/{previous_day}/pages` as old
                full join `//home/statkey/moneymap/snapshots_tmp/{report_date}/pages` as new USING(page_id)
            );

            INSERT INTO `{outputPrefix}/{report_date}/monitorings/diff_by_page` WITH TRUNCATE
            SELECT
                *
            from $join_pages
            where   (old_is_internal = true or new_is_internal = true) and
                    (is_eq_abc_oebs_id != 'NO_CHANGES' OR
                    is_eq_abc_id != 'NO_CHANGES' OR
                    is_eq_traffic_type != 'NO_CHANGES' OR
                    is_eq_traffic_sub_type != 'NO_CHANGES' OR
                    is_eq_traffic_sub_sub_type != 'NO_CHANGES' OR
                    is_eq_os != 'NO_CHANGES' OR
                    is_eq_is_internal != 'NO_CHANGES');

            INSERT INTO `{outputPrefix}/{report_date}/monitorings/diff_by_page_delete` WITH TRUNCATE
            SELECT
                *
            from $join_pages
            where   (old_is_internal = true or new_is_internal = true) and
                    (is_eq_abc_oebs_id = 'DELETE' OR
                    is_eq_abc_id = 'DELETE' OR
                    is_eq_traffic_type = 'DELETE' OR
                    is_eq_traffic_sub_type = 'DELETE' OR
                    is_eq_traffic_sub_sub_type = 'DELETE' OR
                    is_eq_os = 'DELETE' OR
                    is_eq_is_internal = 'DELETE');

            $join_blocks = (
                SELECT
                    new.page_id                 as page_id,
                    new.block_id                 as block_id,
                    old.abc_oebs_id             as old_abc_oebs_id,
                    new.abc_oebs_id             as new_abc_oebs_id,
                    old.abc_id                  as old_abc_id,
                    new.abc_id                  as new_abc_id,
                    old.traffic_type            as old_traffic_type,
                    new.traffic_type            as new_traffic_type,
                    old.traffic_sub_type        as old_traffic_sub_type,
                    new.traffic_sub_type        as new_traffic_sub_type,
                    old.traffic_sub_sub_type    as old_traffic_sub_sub_type,
                    new.traffic_sub_sub_type    as new_traffic_sub_sub_type,
                    old.platform                as old_platform,
                    new.platform                as new_platform,
                    old.os                      as old_os,
                    new.os                      as new_os,
                    old.is_internal             as old_is_internal,
                    new.is_internal             as new_is_internal,
                    $eq_fields(old.abc_oebs_id, new.abc_oebs_id) as is_eq_abc_oebs_id,
                    $eq_fields(old.abc_id, new.abc_id) as is_eq_abc_id,
                    $eq_fields(old.traffic_type, new.traffic_type) as is_eq_traffic_type,
                    $eq_fields(old.traffic_sub_type, new.traffic_sub_type) as is_eq_traffic_sub_type,
                    $eq_fields(old.traffic_sub_sub_type, new.traffic_sub_sub_type) as is_eq_traffic_sub_sub_type,
                    $eq_fields(old.os, new.os) as is_eq_os,
                    $eq_fields(old.is_internal, new.is_internal) as is_eq_is_internal
                from `//home/statkey/moneymap/snapshots_tmp/{previous_day}/blocks` as old
                full join `//home/statkey/moneymap/snapshots_tmp/{report_date}/blocks` as new USING(page_id, block_id)
            );

            INSERT INTO `{outputPrefix}/{report_date}/monitorings/diff_by_block` WITH TRUNCATE
            SELECT
                *
            from $join_blocks
            where   (old_is_internal = true or new_is_internal = true) and
                    (is_eq_abc_oebs_id != 'NO_CHANGES' OR
                    is_eq_abc_id != 'NO_CHANGES' OR
                    is_eq_traffic_type != 'NO_CHANGES' OR
                    is_eq_traffic_sub_type != 'NO_CHANGES' OR
                    is_eq_traffic_sub_sub_type != 'NO_CHANGES' OR
                    is_eq_os != 'NO_CHANGES' OR
                    is_eq_is_internal != 'NO_CHANGES');

            INSERT INTO `{outputPrefix}/{report_date}/monitorings/diff_by_block_delete` WITH TRUNCATE
            SELECT
                *
            from $join_blocks
            where   (old_is_internal = true or new_is_internal = true) and
                    (is_eq_abc_oebs_id = 'DELETE' OR
                    is_eq_abc_id = 'DELETE' OR
                    is_eq_traffic_type = 'DELETE' OR
                    is_eq_traffic_sub_type = 'DELETE' OR
                    is_eq_traffic_sub_sub_type = 'DELETE' OR
                    is_eq_os = 'DELETE' OR
                    is_eq_is_internal = 'DELETE');
        """.format(
            previous_day=previous_day,
            report_date=self.Parameters.report_date,
            outputPrefix=self.Parameters.outputPrefix,
            eq_func=eq_func
        )

    def run_query(self, query, title):
        from yql.api.v1.client import YqlClient
        secret = yav.Secret(self.Parameters.YQLSecretID)
        token = secret.data()[self.Parameters.YQLTokenKey]
        client = YqlClient(db='hahn', token=token)
        request = client.query(query, syntax_version=1, title=title)
        request.run()
        if not request.get_results().is_success:
            error_description = '\n'.join([str(err) for err in request.get_results().errors])
            logging.error(error_description)
            raise RuntimeError(error_description)

    def save_pages_report(self, yt_token=None):
        from dominate.util import raw

        pages_generator = HTMLFromYTGenerator("hahn", "//home/statkey/moneymap/new/{report_date}/report_pages"
                                              .format(report_date=self.Parameters.report_date), yt_token)

        def mm_page_mark_link(row):
            return raw('<a target="_blank" rel="noopener noreferrer" href="{link}">{link}</a>'
                       .format(link="https://moneymap.in.yandex-team.ru/pages/{page_id}"
                               .format(**row)))

        self.data.path.joinpath("pages_not_marked.html").write_bytes(
            pages_generator.generate(
                ["page_id", "link", "source"],
                {"link": mm_page_mark_link},
                lambda row: row["is_auto_marked"] is False,
                lambda r1, r2: r1["page_id"] <= r2["page_id"],
                True
            )
        )

        self.data.path.joinpath("pages_marked_no_conflicts.html").write_bytes(
            pages_generator.generate(
                ["page_id", "traffic_type", "traffic_sub_type", "traffic_sub_sub_type", "os", "platform", "abc_id",
                 "abc_oebs_id", "source", "caption", "link"],
                {
                    "caption": lambda row: "?????",
                    "link": mm_page_mark_link
                },
                lambda row: row["is_auto_marked"] is True and row["has_conflicts"] is False,
                lambda r1, r2: r1["page_id"] <= r2["page_id"],
                True
            )
        )

        self.data.path.joinpath("pages_marked_with_conflicts.html").write_bytes(
            pages_generator.generate(
                ["page_id", "abc_id", "abc_oebs_id", "traffic_type", "traffic_sub_type", "traffic_sub_sub_type", "os",
                 "platform", "caption", "source", "link"],
                {
                    "caption": lambda row: "?????",
                    "link": lambda row:
                    raw('<a target="_blank" rel="noopener noreferrer" href="{link}">{link}</a>'
                        .format(link="https://moneymap.yandex-team.ru/conflicts/pages/{page_id}?"
                                     "abc_id={abc_id}&"
                                     "abc_oebs_id={abc_oebs_id}&"
                                     "traffic_type={traffic_type}&"
                                     "traffic_sub_type={traffic_sub_type}&"
                                     "traffic_sub_sub_type={traffic_sub_sub_type}&"
                                     "platform={platform}&os={os}&"
                                     "is_internal=true&mode=resolve_conflicts"
                                .format(**row)))
                },
                lambda row: row["is_auto_marked"] is True and row["has_conflicts"] is True,
                lambda r1, r2: r1["page_id"] <= r2["page_id"],
                True
            )
        )

    def save_blocks_report(self, yt_token=None):
        from dominate.util import raw

        blocks_generator = HTMLFromYTGenerator("hahn", "//home/statkey/moneymap/new/{report_date}/report_blocks"
                                               .format(report_date=self.Parameters.report_date), yt_token)

        def mm_block_mark_link(row):
            return raw('<a target="_blank" rel="noopener noreferrer" href="{link}">{link}</a>'
                       .format(link="https://moneymap.yandex-team.ru/blocks/{page_id}-{block_id}"
                               .format(**row)))

        self.data.path.joinpath("blocks_not_marked.html").write_bytes(
            blocks_generator.generate(
                ["page_id", "block_id", "link"],
                {"link": mm_block_mark_link},
                lambda row: row["is_auto_marked"] is False,
                lambda r1, r2: (r1["page_id"], r1["block_id"]) <= (r2["page_id"], r2["block_id"]),
                True
            )
        )

        self.data.path.joinpath("blocks_marked_no_conflicts.html").write_bytes(
            blocks_generator.generate(
                ["page_id", "traffic_type", "traffic_sub_type", "traffic_sub_sub_type", "os", "platform", "abc_id",
                 "abc_oebs_id", "source", "caption", "link"],
                {
                    "caption": lambda row: "?????",
                    "link": mm_block_mark_link
                },
                lambda row: row["is_auto_marked"] is True and row["has_conflicts"] is False,
                lambda r1, r2: (r1["page_id"], r1["block_id"]) <= (r2["page_id"], r2["block_id"]),
                True
            )
        )

        self.data.path.joinpath("blocks_marked_with_conflicts.html").write_bytes(
            blocks_generator.generate(
                ["page_id", "abc_id", "abc_oebs_id", "traffic_type", "traffic_sub_type", "traffic_sub_sub_type", "os",
                 "platform", "caption", "source", "link"],
                {
                    "caption": lambda row: "?????",
                    "link": lambda row:
                    raw('<a target="_blank" rel="noopener noreferrer" href="{link}">{link}</a>'
                        .format(link="https://moneymap.yandex-team.ru/conflicts/blocks/{page_id}-{block_id}?"
                                     "abc_id={abc_id}&"
                                     "abc_oebs_id={abc_oebs_id}&"
                                     "traffic_type={traffic_type}&"
                                     "traffic_sub_type={traffic_sub_type}&"
                                     "traffic_sub_sub_type={traffic_sub_sub_type}&"
                                     "platform={platform}&os={os}&"
                                     "is_internal=true&mode=resolve_conflicts"
                                .format(**row)))
                },
                lambda row: row["is_auto_marked"] is True and row["has_conflicts"] is True,
                lambda r1, r2: (r1["page_id"], r1["block_id"]) <= (r2["page_id"], r2["block_id"]),
                True
            )
        )

    def create_ticket(self, startrek_client, links):
        issue = startrek_client.issues.create(
            queue='MONEYMAPRESOLVE',
            assignee=self.Parameters.TicketAssignee,
            summary='Появились неразмеченные сущности {}'.format(self.Parameters.report_date),
            type={'name': 'Task'},
            description="""
            Неразмеченные пейджи: {}


            Неразмеченные блоки: {}
            Размеченные пейджи: {}
            Размеченные блоки: {}

            Табличка main_page {}
            """.format(*links)
        )
        issue.comments.create(text="""Нужно вручную разметить пейджи {}""".format(links[0]), summonees=[self.Parameters.TicketAssignee])

    def on_execute(self):
        self.run_query(self.prepare_new_entities_query(), "MM: new_entities | YQL")
        self.run_query(self.not_marked_pages_by_domain(), "MM: not marked by domain | YQL")
        self.run_query(self.prepare_alice_query(), "MM: alice | YQL")
        self.run_query(self.prepare_vh_query(), "MM: vh | YQL")
        self.run_query(self.prepare_zen_page_query(), "MM: zen pages | YQL")
        self.run_query(self.prepare_main_search(), "MM: main_search | YQL")
        self.run_query(self.prepare_not_marked_mm_pages(), "MM: not_marked_mm_pages | YQL")
        self.run_query(self.prepare_not_marked_mm_blocks(), "MM: not_marked_mm_blocks | YQL")
        self.run_query(self.prepare_pages_report(), "MM: pages_report | YQL")
        self.run_query(self.prepare_blocks_report(), "MM: blocks_report | YQL")
        self.run_query(self.diff_day_by_day(), "MM: pages diff between 2 days | YQL")

        resource = MoneymapReport(self, "html tables with MoneyMap reports", "mm_reports")
        self.data = sdk2.ResourceData(resource)
        self.data.path.mkdir(0o755)
        yt_token = yav.Secret(self.Parameters.YQLSecretID).data()[self.Parameters.YQLTokenKey]
        self.save_pages_report(yt_token)
        self.save_blocks_report(yt_token)

        from startrek_client import Startrek
        import yt.wrapper as yt

        yt.config['token'] = yt_token
        yt.config.set_proxy('hahn')
        if(yt.row_count("//home/statkey/moneymap/new/{report_date}/report_pages".format(report_date=self.Parameters.report_date)) > 0):
            conflict_count = 0
            for raw in yt.read_table("//home/statkey/moneymap/new/{report_date}/report_pages".format(report_date=self.Parameters.report_date)):
                if (raw["has_conflicts"] or raw["abc_id"] is None):
                    conflict_count = conflict_count + 1
            if conflict_count > 0:
                startrek_client = Startrek(useragent='', token=yav.Secret(self.Parameters.StartrekTokenID).data()['token'])
                make_link = lambda name : 'https://proxy.sandbox.yandex-team.ru/{}/{}'.format(resource.id, name)
                self.create_ticket(startrek_client, [
                    make_link("pages_not_marked.html"),
                    make_link("blocks_not_marked.html"),
                    make_link("pages_marked_with_conflicts.html"),
                    make_link("blocks_marked_with_conflicts.html"),
                    "https://yt.yandex-team.ru/hahn/navigation?path=//home/statkey/moneymap/new/{report_date}/"
                    "not_marked_pages/main_search".format(report_date=self.Parameters.report_date)
                ])
