# coding=utf-8
from __future__ import unicode_literals

import logging

import json
import os
import signal

from sandbox import sdk2
from sandbox.sdk2.helpers import gdb
from sandbox.common.types.task import Semaphores
from sandbox.projects.common.arcadia import sdk as arcadiasdk

import sandbox.projects.sandbox.resources as sb_resources


def maybe_decode_json(obj):
    # sdk2.parameters.JSON это String с модификатором Json, поэтому
    # при создании из API Реактора происходит путаница – можно создать
    # только закодированный JSON в виде строки. Тут мы пробуем превратить
    # строку в разобранный JSON, если она содержит валидный json.

    if not isinstance(obj, basestring):
        return obj
    else:
        try:
            return json.loads(obj)
        except ValueError:
            return obj


class cached_property(object):
    def __init__(self, func):
        self._func = func

    def __get__(self, obj, owner):
        assert obj is not None, 'call {} on an instance'.format(self._func.__name__)
        answer = obj.__dict__[self._func.__name__] = self._func(obj)
        return answer


class StatkeySimpleTask(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 8000
        cores = 1
        # dns = ctm.DnsType.DNS64  # copied from STATINFRA_TASK
        ram = 8 * 1024
        # ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 2048, None)  # copied from STATINFRA_TASK

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        resource = sdk2.parameters.Resource('Реcурс с релизом. Содержимое будет доступно через ./resource/')
        container = sdk2.parameters.Resource('Ресурс с контейнером LXC', resource_type=sb_resources.LXC_CONTAINER)
        mount_arcadia_url = sdk2.parameters.String('arcadia url, содержимое будет доступно через ./arcadia/',
                                                   description='e.g. arcadia-arc:/#trunk ; Также нужно указать секрет ARC_TOKEN')

        secrets = sdk2.parameters.JSON(r'Словарь с указанием куда класть секреты: {ENV_VAR_NAME: {id: SECRET_ID, key: SECRET_KEY}}', default={})
        argv = sdk2.parameters.JSON('Команда для запуска', default=[])
        custom_script = sdk2.parameters.String("Shell script to execute instead of argv command", multiline=True)
        env = sdk2.parameters.JSON('Дополнительные переменные окружения')
        semaphores = sdk2.parameters.JSON('Список семафоров: список имён, либо словарь {name,weight?,capacity?}')

        use_output_parameter = sdk2.parameters.Bool('Писать ли stdout в параметр output. По-умолчанию – нет. В stdout должен быть записан json', default=False)

        custom_kill_timeout = sdk2.parameters.Integer('Максимальное время работы задачи в секундах', default=10*3600)

        with sdk2.parameters.Output:
            output = sdk2.parameters.JSON('Результат работы задачи')

    def _create_symlink(self, src, dst):
        if os.path.exists(dst):
            if os.path.islink(dst) and os.readlink(dst) == src:
                logging.info('Symlink %s -> %s already exists' % (dst, src))
            else:
                raise RuntimeError('Cant create symlink: file %s already exists' % src)
        else:
            os.symlink(src, dst)

    def _create_symlinks(self):
        if self.Parameters.resource is not None:
            self._create_symlink(
                sdk2.ResourceData(self.Parameters.resource).path.as_posix(),
                './resource'
            )

    @cached_property
    def _secrets(self):
        unique_secrets = set()

        for _, sec in self.Parameters.secrets.items():
            unique_secrets.add(sec['id'])

        secrets = sdk2.yav.Yav(**{
            id: sdk2.yav.Secret(id) for id in unique_secrets
        })

        answer = {}

        for name, sec in self.Parameters.secrets.items():
            answer[name] = getattr(secrets, sec['id'])[sec['key']]

        return answer

    def _prepare_env(self):
        env = {}

        if self.Parameters.env:
            logging.info('update env: %r' % self.Parameters.env)
            env.update(self.Parameters.env)

        env.update(self._secrets)

        return env

    def _set_semaphores(self):
        if not self.Parameters.semaphores:
            return

        semaphores = []
        for sem in self.Parameters.semaphores:
            if isinstance(sem, basestring):
                semaphores.append(Semaphores.Acquire(name=sem))
            elif isinstance(sem, dict):
                semaphores.append(Semaphores.Acquire(**sem))
            else:
                raise ValueError('Wrong type of semaphore parameter: %r' % sem)

        self.Requirements.semaphores = Semaphores(acquires=semaphores)

    def _set_container(self):
        if self.Parameters.container:
            self.Requirements.container_resource = self.Parameters.container

    def on_save(self):
        self.Parameters.env = maybe_decode_json(self.Parameters.env)
        self.Parameters.argv = maybe_decode_json(self.Parameters.argv)
        self.Parameters.semaphores = maybe_decode_json(self.Parameters.semaphores)
        self.Parameters.secrets = maybe_decode_json(self.Parameters.secrets)
        self.Parameters.resource = self.Parameters.resource or None  # replace 0 with None, because its a hard to put None here from Reactor
        self.Parameters.container = self.Parameters.container or None  # replace 0 with None, because its a hard to put None here from Reactor

        assert not (bool(self.Parameters.argv) and bool(self.Parameters.custom_script)), "Specify only one: argv or custom_script"

        self._set_semaphores()
        self._set_container()

        self.Parameters.kill_timeout = self.Parameters.custom_kill_timeout

    def _call_subprocess(self, stdout, stderr):
        if self.Parameters.custom_script:
            with open('./custom_script.sh', 'w') as file:
                file.write(self.Parameters.custom_script)
            argv = ["bash", "./custom_script.sh"]
        else:
            argv = self.Parameters.argv

        if self.Parameters.use_output_parameter:
            return sdk2.helpers.subprocess.check_output(argv, env=self._prepare_env(), stderr=stderr)
        else:
            sdk2.helpers.subprocess.check_call(argv, env=self._prepare_env(), stderr=stderr, stdout=stdout)
            return None

    def _execute(self):
        with sdk2.helpers.ProcessLog(self, logger='run') as log:
            if not self.Parameters.use_output_parameter:
                self.set_info(
                    gdb.get_html_view_for_logs_file('stdout', log.stdout.path.relative_to(self.log_path()), self.log_resource),
                    do_escape=False
                )
            self.set_info(
                gdb.get_html_view_for_logs_file('stderr', log.stderr.path.relative_to(self.log_path()), self.log_resource),
                do_escape=False
            )

            if self.Parameters.mount_arcadia_url:
                arc_oauth_token = self._secrets['ARC_TOKEN']
                with arcadiasdk.mount_arc_path(self.Parameters.mount_arcadia_url, arc_oauth_token=arc_oauth_token) as mount_point:
                    self._create_symlink(mount_point, './arcadia')
                    output = self._call_subprocess(log.stdout, log.stderr)
            else:
                output = self._call_subprocess(log.stdout, log.stderr)

        if self.Parameters.use_output_parameter:
            self.Parameters.output = json.loads(output)

    def on_execute(self):
        self._create_symlinks()
        self._execute()

    def on_before_timeout(seconds):  # запускается за seconds секунд до таймаута
        # по умолчанию запускается за [10, 30, 60, 60 * 3, 60 * 5] до жёсткого SIGKILL
        if seconds <= 3 * 60:
            for process in sdk2.helpers.ProcessRegistry:
                try:
                    os.kill(process.pid, signal.SIGTERM)
                except OSError:
                    continue
