# coding=utf-8
from __future__ import unicode_literals

import logging

import json
import os
import re

from sandbox import sdk2
from sandbox.projects.statkey.BuildCalculationResource import CalculationResource
from sandbox.sdk2.helpers import gdb


def maybe_decode_json(obj):
    # sdk2.parameters.JSON это String с модификатором Json, поэтому
    # при создании из API Реактора происходит путаница – можно создать
    # только закодированный JSON в виде строки. Тут мы пробуем превратить
    # строку в разобранный JSON, если она содержит валидный json.

    if not isinstance(obj, basestring):
        return obj
    else:
        try:
            return json.loads(obj)
        except ValueError:
            return obj


class cached_property(object):
    def __init__(self, func):
        self._func = func

    def __get__(self, obj, owner):
        assert obj is not None, 'call {} on an instance'.format(self._func.__name__)
        answer = obj.__dict__[self._func.__name__] = self._func(obj)
        return answer


class StatkeyRun(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 8000
        cores = 1
        # dns = ctm.DnsType.DNS64  # copied from STATINFRA_TASK
        ram = 8 * 1024
        # ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 2048, None)  # copied from STATINFRA_TASK

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        resource = sdk2.parameters.Resource('Реcурс с релизом', resource_type=CalculationResource, required=True)

        with sdk2.parameters.RadioGroup('Режим', required=True) as mode:
            mode.values['run'] = mode.Value(value='run', default=True)
            mode.values['info'] = mode.Value(value='info')

        dates = sdk2.parameters.String('dates', required=True)

        spec = sdk2.parameters.JSON('Произвольные входные параметры: type, secrets, proxy, scale, argv',
                                    default={'secrets': {'YAV_TOKEN': {"uuid": 'sec-01eme6xnyx1ed60zjfa9e6t9j4', "key": "token"}}})

        with sdk2.parameters.Output:
            info = sdk2.parameters.JSON('Результат ручки info')

    def on_enqueue(self):
        self.Parameters.spec = maybe_decode_json(self.Parameters.spec)

    def fetch_secrets(self):
        unique_secrets = set()

        for _, secret_spec in self.Parameters.spec['secrets'].items():
            id = secret_spec.get('uuid', secret_spec.get('id'))  # FIXME `id` is old variant
            unique_secrets.add(id)

        secrets = sdk2.yav.Yav(**{
            id: sdk2.yav.Secret(id) for id in unique_secrets
        })

        answer = {}

        for name, secret_spec in self.Parameters.spec['secrets'].items():
            id = secret_spec.get('uuid', secret_spec.get('id'))  # FIXME `id` is old variant
            key = secret_spec['key']
            answer[name] = getattr(secrets, id)[key]

        return answer

    def on_execute(self):
        if self.Parameters.spec['type'] == 'nile_over_yql':
            executor = NileOverYQLExecutor(self)
        elif self.Parameters.spec['type'] == 'nile_over_yt':
            executor = NileOverYTExecutor(self)
        elif self.Parameters.spec['type'] == 'yql':
            executor = YQLExecutor(self)
        elif self.Parameters.spec['type'] == 'custom':
            executor = CustomExecutor(self)
        elif self.Parameters.spec['type'] == 'runplan':
            executor = RunplanExecutor(self)
        else:
            raise NotImplementedError

        if self.Parameters.mode == 'run':
            executor.run()
        elif self.Parameters.mode == 'info':
            info = executor.info()
            assert 'input' in info
            assert 'output' in info
            self.Parameters.info = info


class ExecutorBase(object):
    def __init__(self, task):
        logging.info("Initialize {}", type(self))
        self._task = task

        self.validate()

    @property
    def spec(self):
        return self._task.Parameters.spec

    @property
    def dates(self):
        return self._task.Parameters.dates

    def validate(self):
        logging.info("Skip validation")

    def env(self):
        env = {}

        env.update(self._task.fetch_secrets())

        return env

    def _create_symlink(self):
        src = sdk2.ResourceData(self._task.Parameters.resource).path.as_posix()
        dst = './resource'

        if os.path.exists(dst):
            if os.path.islink(dst) and os.readlink(dst) == src:
                logging.info('Symlink %s -> %s already exists' % (dst, src))
            else:
                raise RuntimeError('Cant create symlink: file %s already exists' % src)
        else:
            os.symlink(src, dst)

    def _execute(self, argv, return_output=False):
        self._create_symlink()

        with sdk2.helpers.ProcessLog(self._task, logger='run-binary') as log:
            for stream in ['out', 'err']:
                self._task.set_info(
                    gdb.get_html_view_for_logs_file(
                        'Std' + stream, getattr(log, 'std' + stream).path.relative_to(self._task.log_path()), self._task.log_resource
                    ),
                    do_escape=False
                )
                # TODO добавить сюда в set_info побольше ссылок

            if return_output:
                return sdk2.helpers.subprocess.check_output(argv, env=self.env(), stderr=log.stderr)
            else:
                sdk2.helpers.subprocess.check_call(argv, env=self.env(), stderr=log.stderr, stdout=log.stdout)
                return


class NileExecutorBase(ExecutorBase):
    def validate(self):
        assert 'scale' in self.spec
        assert 'proxy' in self.spec

    def info(self):
        nile_info = json.loads(self.nile_info())
        info = {'input': {'yt': []}, 'output': {'yt': []}}
        for table in nile_info['source_tables']:
            info['input']['yt'].append('{proxy}:{path}'.format(proxy=self.spec['proxy'], path=table))
        for table in nile_info['destination_tables']:
            info['output']['yt'].append('{proxy}:{path}'.format(proxy=self.spec['proxy'], path=table))
        return info


class NileOverYQLExecutor(NileExecutorBase):
    def validate(self):
        super(NileOverYQLExecutor, self).validate()
        assert 'secrets' in self.spec
        assert 'YQL_TOKEN' in self.spec['secrets']
        assert 'YT_TOKEN' in self.spec['secrets']

    def run(self):
        self._execute([
            'resource/bin/run_python_udf',
            'resource/bin/udf.so',
            'run',
            '--proxy', self.spec['proxy'],
            '--dates', self.dates,
            '--scale', self.spec['scale'],
            '--use-yql'
        ])

    def nile_info(self):
        return self._execute(
            [
                'resource/bin/run_python_udf',
                'resource/bin/udf.so',
                'info',
                '--proxy', self.spec['proxy'],
                '--dates', self.dates,
                '--scale', self.spec['scale'],
                '--use-yql'
            ],
            return_output=True
        )


class NileOverYTExecutor(ExecutorBase):
    def validate(self):
        super(NileOverYQLExecutor, self).validate()
        assert 'secrets' in self.spec
        assert 'YT_TOKEN' in self.spec['secrets']

    def run(self):
        self._execute([
            'resource/bin/job',
            'run',
            '--proxy', self.spec['proxy'],
            '--dates', self.dates,
            '--scale', self.spec['scale'],
        ])

    def nile_info(self):
        return self._execute(
            [
                'bin/job',
                'info',
                '--proxy', self.spec['proxy'],
                '--dates', self.dates,
                '--scale', self.spec['scale'],
            ],
            return_output=True
        )


class YQLExecutor(ExecutorBase):
    def validate(self):
        assert 'scale' in self.spec
        assert 'proxy' in self.spec
        assert self._task.Parameters.mode == 'run'  # TODO info mode
        assert 'secrets' in self.spec
        assert 'YQL_TOKEN' in self.spec['secrets']

    def __init__(self, task):
        super(YQLExecutor, self).__init__(task)
        self._execute([
            'bash',
            '-c',
            'curl -Lo yql https://yql.yandex.net/download/cli/linux/yql && chmod +x yql'
        ])

    def run(self):
        self._execute([
            'bash', '-c',
            'cat resource/query.sql | ./yql --syntax-version=1 --parameter-proxy="{proxy}" --parameter-scale="{scale}" --parameter-dates="{dates}"'.format(
                proxy=self.spec['proxy'],
                dates=self.dates,
                scale=self.spec['scale'],
            ),
        ])

    def info(self):
        raise NotImplementedError


class CustomExecutor(ExecutorBase):
    def validate(self):
        assert 'argv' in self._task.Parameters.spec
        assert self._task.Parameters.mode == 'run'

    def run(self):
        return self._execute(self.spec['argv'])

    def info(self):
        raise NotImplementedError


class RunplanExecutor(ExecutorBase):
    def validate(self):
        assert re.match(r'^[0-9]{4}-[0-9]{2}-[0-9]{2}(T[0-9]{2}:[0-9]{2}:[0-9]{2})?$', self.dates), "Dates should be in format YYYY-MM-DD[THH:MM:SS]"

    def info(self):
        output = self._execute(
            [
                'resource/bin/job',
                'plan',
                self.dates
            ],
            return_output=True
        )
        loaded = json.loads(output)
        assert isinstance(loaded, list) and len(loaded) == 1, "Output of job info (plan) should be a list of one element"
        return loaded[0]

    def run(self):
        self._execute([
            'resource/bin/job',
            'run',
            json.dumps({"user_time": self.dates})
        ])
