# coding: utf-8

from datetime import datetime, timedelta
from textwrap import dedent

import requests
import lxml.html

from sandbox import sdk2
from sandbox.projects.common import task_env


def send_tg_message(text, *other, bot_token, chat_id):
    resp = requests.post(
        f'https://api.telegram.org/bot{bot_token}/sendMessage',
        {
            'chat_id': chat_id,
            'parse_mode': 'HTML',
            'disable_web_page_preview': True,
            'text': ''.join([text, *other]).strip(),
        }
    )

    resp.raise_for_status()
    return resp


def send_tg_event(event, *, bot_token, chat_id):
    return send_tg_message(
        dedent(f'''
             <b>{event['summary']}</b>
             <i>{event['start'].time()} — {event['end'].time()}</i>
        '''),
        dedent(event['description'] or ''),
        bot_token=bot_token,
        chat_id=chat_id
    )


def get_events(private_token, from_, to):
    resp = requests.get(
        'https://api.calendar.yandex-team.ru/export/html.xml',
        {
            'private_token': private_token,
            'tz_id': 'Europe/Moscow',
            'from': str(from_.date()),
            'to': str(to.date()),
            'lang': 'en',
        }
    )

    resp.raise_for_status()

    answer = []

    for item in lxml.html.fromstring(resp.text).xpath('//div[@class="b-content-event"]'):
        start, end = item.findall('div[@class="e-time"]/span')
        summary = item.find('h1')
        description = item.find('div[@class="e-description"]')
        answer.append({
            'summary': summary.text_content(),
            'description': description.text_content(),
            'start': datetime.strptime(start.text_content(), '%d %B %Y %H:%M'),
            'end': datetime.strptime(end.text_content(), '%d %B %Y %H:%M'),
        })

    return [e for e in answer if from_ <= e['start'] <= to]


def round_5min(dt):
    date = dt.date()
    time = dt.time()
    time = time.replace(minute=time.minute // 5 * 5, second=0, microsecond=0)
    return datetime.combine(date, time)


class TelegramCalendarNotifier(sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(sdk2.Parameters):
        secret = sdk2.parameters.YavSecret(
            "YAV secret with telegram token, chat_id and calendar link (TODO better description)",
            required=True
        )
        timedelta = sdk2.parameters.Integer('Consider events in [dt, dt + timedelta], seconds', default=15*60)
        trunc_now = sdk2.parameters.Bool('Truncate datetime.now() to 5min', default=True)

    def on_execute(self):
        secret = self.Parameters.secret.data()
        now = datetime.now()
        if self.Parameters.trunc_now:
            now = round_5min(now)

        for event in get_events(secret['calendar'], now, now + timedelta(seconds=self.Parameters.timedelta)):
            send_tg_event(event, bot_token=secret['telegram'], chat_id=secret['chat_id'])
