# -*- coding: utf-8 -*-

import logging
import os
import tarfile
import requests
import time


from sandbox import sdk2
from sandbox.common import fs
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import gnupg, debpkg
from sandbox.sdk2 import ssh

from ..utils.conductor import ConductorAPIClient
from ..utils.helpers import (
    find_file_by_glob, make_html_link, generate_dupload_config
)
from ..utils.project_builder import (
    StatlibsProjectBuilder, PY2_BINARY, PY3_BINARY, PY3_PACKAGE_PREFIX,
    BIONIC_CONTAINER_RESOURCE, XENIAL_CONTAINER_RESOURCE,
)
from ..utils.resources import STATLIBS_PACKAGE_TARBALL


KEYS_OWNER = 'STATINFRA'
KEYS_USER = 'robot-statinfra'

GPG_SECRET_KEY_NAME = '{}-gpg-private'.format(KEYS_USER)
GPG_PUBLIC_KEY_NAME = '{}-gpg-public'.format(KEYS_USER)
GPG_KEY_ID = KEYS_USER

SSH_KEY_NAME = '{}-ssh'.format(KEYS_USER)

CONDUCTOR_KEY_NAME = '{}-conductor-auth'.format(KEYS_USER)

CONDUCTOR_BRANCHES = (
    'testing',
    'stable',
    'hotfix',
)

PY2_NON_UNIVERSAL_REPOS = (
    'yandex-precise',
    'yandex-trusty',
    'yandex-xenial',
)

PY36_NON_UNIVERSAL_REPOS = (
    'yandex-bionic',
)

UNIVERSAL_REPOS = (
    'statbox-common',
    'common',
)


class StatlibsDebuilder(StatlibsProjectBuilder):
    """Build & upload debian package for one of statbox projects in Arcadia"""

    class Requirements(StatlibsProjectBuilder.Requirements):
        privileged = True

    class Parameters(StatlibsProjectBuilder.Parameters):
        with sdk2.parameters.Group('Debuild settings') as debuild_settings:
            with sdk2.parameters.String('Python binary', required=True) as python_binary:
                python_binary.values[PY2_BINARY] = python_binary.Value(PY2_BINARY, default=True)
                python_binary.values[PY3_BINARY] = PY3_BINARY

        with sdk2.parameters.Group('Release settings') as release_settings:
            upload_package = sdk2.parameters.Bool('Upload package', default=False)
            with upload_package.value[True]:
                create_ticket = sdk2.parameters.Bool('Create conductor ticket', default=False)
                with create_ticket.value[True]:
                    with sdk2.parameters.String('Release branch', default=CONDUCTOR_BRANCHES[0]) as conductor_branch:
                        for branch in CONDUCTOR_BRANCHES:
                            conductor_branch.values[branch] = branch

                    conductor_projects = sdk2.parameters.List('Conductor project(s)', default=['statbox'])

                use_universal_repo = sdk2.parameters.Bool('Upload to universal repo', default=False)
                with use_universal_repo.value[True]:
                    with sdk2.parameters.String('Universal repo', default=UNIVERSAL_REPOS[0]) as universal_repo:
                        for repo in UNIVERSAL_REPOS:
                            universal_repo.values[repo] = repo

            with create_ticket.value[False]:
                dmove_manually = sdk2.parameters.Bool('Dmove package', default=False)
                with dmove_manually.value[True]:
                    dmove_from_branch = sdk2.parameters.String('From branch', default='unstable')
                    dmove_to_branch = sdk2.parameters.String('To branch', default='stable')

            dep_install = sdk2.parameters.Bool('Install dependencies', default=False)

    @property
    def is_py3_build(self):
        python_binary = self.Parameters.python_binary
        if python_binary not in (PY2_BINARY, PY3_BINARY):
            raise RuntimeError('Unexpected python_binary {}'.format(python_binary))

        return python_binary == PY3_BINARY

    @property
    def binary_package(self):
        py3_candidates = [
            _ for _ in self.Context.binary_packages
            if _.startswith(PY3_PACKAGE_PREFIX)
        ]
        py2_candidates = list(set(self.Context.binary_packages) - set(py3_candidates))
        assert len(py3_candidates) <= 1
        assert len(py2_candidates) <= 1

        if self.is_py3_build:
            if not py3_candidates:
                raise RuntimeError('No python3-* binary packages were found')

            return py3_candidates[0]
        else:
            if not py2_candidates:
                raise RuntimeError('No binary packages for python2 were found')

            return py2_candidates[0]

    def on_save(self):
        if self.is_py3_build:
            self.Requirements.container_resource = BIONIC_CONTAINER_RESOURCE
        else:
            # on bionic we build incompatible with precise package
            # (something wrong with archives extension)
            self.Requirements.container_resource = XENIAL_CONTAINER_RESOURCE

    def do_build(self):
        logging.info('Start build actions')

        self.set_info('Binary package name: {}'.format(self.binary_package))

        if self.Parameters.dep_install:
            logging.info('Installing build dependencies...')
            self.run_process(
                ['apt', '-y', 'update'],
                cwd=self.project_dir,
                logger='dep_install',
            )
            self.run_process(
                ['apt', '-y', 'install', 'pbuilder', 'debootstrap', 'devscripts', 'aptitude'],
                cwd=self.project_dir,
                logger='dep_install',
            )
            self.run_process(
                ['/usr/lib/pbuilder/pbuilder-satisfydepends'],
                cwd=self.project_dir,
                logger='dep_install',
            )

        env = os.environ.copy()
        debuild_cmd = ['debuild']
        if self.is_py3_build:
            env['PY3_BINARY'] = self.Parameters.python_binary
            debuild_cmd.extend(['-e', 'PY3_BINARY'])

        debuild_cmd.extend(['-b', '-k{}'.format(GPG_KEY_ID)])

        logging.info('Running debuild...')
        with gnupg.GpgKey2(KEYS_OWNER, GPG_SECRET_KEY_NAME, GPG_PUBLIC_KEY_NAME):
            self.run_process(
                debuild_cmd,
                cwd=self.debian_home,
                env=env,
                logger='debuild',
            )

        self.set_info('Package was successfully built')

    def do_postbuild(self):
        self.make_resource()

        if self.Parameters.upload_package:
            self.dupload()

            if self.Parameters.create_ticket:
                if self.is_py3_build:
                    raise NotImplementedError(
                        'Creating conductor ticket is not supported for python3'
                    )

                self.create_conductor_ticket()

        if self.Parameters.dmove_manually:
            self.dmove()

    def make_resource(self):
        binary_package = self.binary_package
        source_package = self.Context.source_package
        version = self.Context.build_version
        binary_prefix = '{}_{}'.format(binary_package, version)
        source_prefix = '{}_{}'.format(source_package, version)
        tarball_path = os.path.abspath('{}.tar.gz'.format(binary_prefix))

        logging.info('Creating resource for %s=%s...', binary_package, version)
        logging.debug('Tarball path: %s', tarball_path)

        with fs.WorkDir(os.path.dirname(self.debian_home)):
            deb_file = find_file_by_glob(binary_prefix, '.deb')
            changes_file = find_file_by_glob(source_prefix, '.changes')

            with tarfile.open(tarball_path, mode='w:gz') as tarball:
                tarball.add(deb_file, arcname=os.path.basename(deb_file))
                tarball.add(changes_file, arcname=os.path.basename(changes_file))

        logging.debug('Tarball size: %s', os.stat(tarball_path).st_size)

        tarball_resource = STATLIBS_PACKAGE_TARBALL(
            self,
            description='Package {}={}'.format(binary_package, version),
            path=tarball_path,
            # attributes
            ttl=30,
            package=binary_package,
            version=version,
        )

        sdk2.ResourceData(tarball_resource).ready()

        self.Context.tarball_resource = tarball_resource.id

        html_link = lb.resource_link(tarball_resource.id)

        self.set_info('Package tarball: {}'.format(html_link), do_escape=False)

    def dupload(self):
        logging.info('Preparing package uploading...')

        dupload_config = generate_dupload_config(
            UNIVERSAL_REPOS + PY2_NON_UNIVERSAL_REPOS + PY36_NON_UNIVERSAL_REPOS,
            KEYS_USER,
        )

        if self.Parameters.use_universal_repo:
            repos_list = (self.Parameters.universal_repo,)
        elif self.is_py3_build:
            repos_list = PY36_NON_UNIVERSAL_REPOS
        else:
            repos_list = PY2_NON_UNIVERSAL_REPOS

        logging.debug(' - repos: %s', repos_list)

        logging.info('Uploading package to repo(s)...')
        with ssh.Key(self, KEYS_OWNER, SSH_KEY_NAME):
            with fs.WorkDir(self.debian_home):
                with debpkg.DebRelease(dupload_config) as deb:
                    for repo in repos_list:
                        deb.debrelease(['--to', repo])

        self.set_info(
            'Package was successfully uploaded to: {}'
            .format(', '.join(repos_list))
        )

    def create_conductor_ticket(self):
        logging.info('Creating conductor ticket...')

        client = ConductorAPIClient(
            auth_cookie=sdk2.Vault.data(KEYS_OWNER, CONDUCTOR_KEY_NAME)
        )

        ticket_url = None

        try:
            ticket_url = client.ticket_add(
                self.binary_package,
                self.Context.build_version,
                self.Parameters.conductor_branch,
                projects=list(self.Parameters.conductor_projects) or None,
                comment=self.Context.build_changes,
            )
        except requests.RequestException:
            logging.exception('Conductor API error:')
            self.set_info('Failed to create conductor ticket')
            return

        if ticket_url is None:
            html_link = '&ltNO LINK&gt'
        else:
            html_link = make_html_link(ticket_url)

        self.set_info('Conductor ticket: {}'.format(html_link), do_escape=False)

    def dmove(self):
        logging.info('Dmove package')

        if self.Parameters.upload_package:
            # I believed to some random guy here
            # https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/BuildNannyDeb/__init__.py?rev=r3997112#L115-118
            # who wrote that dupload changes can be not reindexed yet if we dmove right after dupload.
            # TODO: understand that this sleep is really needed.
            time.sleep(2 * 60)

        if self.Parameters.use_universal_repo:
            repos_list = (self.Parameters.universal_repo,)
        elif self.is_py3_build:
            repos_list = PY36_NON_UNIVERSAL_REPOS
        else:
            repos_list = PY2_NON_UNIVERSAL_REPOS

        logging.debug(' - repos: %s', repos_list)

        logging.info('Dmove package in repos...')
        with ssh.Key(self, KEYS_OWNER, SSH_KEY_NAME):
            for repo in repos_list:
                dmove_cmd = 'ssh {user}@dupload.dist.yandex.ru sudo dmove {repo} {to_branch} {package} {version} {from_branch}'.format(
                    user=KEYS_USER,
                    repo=repo,
                    to_branch=self.Parameters.dmove_to_branch,
                    package=self.binary_package,
                    version=self.Context.build_version,
                    from_branch=self.Parameters.dmove_from_branch,
                ).split()

                self.set_info('Execute {!r}'.format(dmove_cmd))

                self.run_process(
                    dmove_cmd,
                    logger='dmove',
                )

        self.set_info('All dmoves finished successfully')
