# -*- coding: utf-8 -*-

import logging
import re

import requests


def hide_sensitive_data(string, symbols_to_show=3):
    if string is None:
        return None

    return '{:*<{n}}'.format(string[:symbols_to_show], n=len(string))


class ConductorAPIClient(object):
    def __init__(
        self,
        host='c.yandex-team.ru',
        protocol='https',
        auth_token=None,
        auth_cookie=None,
    ):
        logging.info('Initializing conductor client...')
        logging.debug(' - host: %s', host)
        logging.debug(' - protocol: %s', protocol)
        logging.debug(' - auth token: %s', hide_sensitive_data(auth_token))
        logging.debug(' - auth cookie: %s', hide_sensitive_data(auth_cookie))

        self._host = host
        self._protocol = protocol
        self._auth_token = auth_token
        self._auth_cookie = auth_cookie

    @property
    def _headers(self):
        if self._auth_token is not None:
            return {'Authorization': 'OAuth {}'.format(self._auth_token)}

        if self._auth_cookie is not None:
            return {'Cookie': 'conductor_auth={}'.format(self._auth_cookie)}

        return None

    def _request(self, method, uri, **kws):
        url = '{protocol}://{host}/{uri}'.format(
            protocol=self._protocol,
            host=self._host,
            uri=uri,
        )

        logging.info('Requesting conductor API...')
        logging.debug(' - method %s', method)
        logging.debug(' - url: %s', url)
        logging.debug(' - params: %s', kws)
        response = requests.request(method, url, headers=self._headers, **kws)
        response.raise_for_status()

        return response

    def _post(self, uri, **kws):
        return self._request('POST', uri, **kws)

    def whoami(self):
        return self._request('GET', 'auth_update/who_am_i').content

    def ticket_add(self, package, version, branch, projects=None, comment=None):
        json_data = {
            'ticket': {'branch': branch},
            'package': {'0': package},
            'version': {'0': version},
        }

        if projects is not None:
            json_data['filters'] = {'projects': projects}

        if comment is not None:
            json_data['ticket']['comment'] = comment

        content = self._post('auth_update/ticket_add', json=json_data).content
        logging.debug('Response content: %r', content)

        if content:
            match = re.search(r'URL:\s(\S+)', content)
            if match:
                return match.group(1)

        return None
