# -*- coding: utf-8 -*-

import json
import logging
import os

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.vcs import arc

from .task_subprocess import run_task_subprocess


SUPPORTED_PROJECTS = (
    ('Nile', 'statbox/nile'),
    ('Nile Debug', 'statbox/nile_debug'),
    ('QB2 Core', 'statbox/qb2_core'),
    ('QB2', 'statbox/qb2'),
    ('Statface Client', 'library/python/statface_client'),
    ('Clickhouse Client', 'library/python/clickhouse_client'),
    ('STEP client', 'statbox/step_client'),
    ('Python Statinfra', 'statbox/python-statinfra'),
    ('Statbox Bindings', 'statbox/libstatbox/python'),
    ('Signurl Bindings', 'statbox/bindings/signurl'),
    ('Metrika Bindings', 'statbox/bindings/metrikatraficsource'),
    ('Cyson Bindings', 'statbox/bindings/cyson'),
    ('Tdigest Bindings', 'statbox/bindings/tdigest'),
    ('Common Config', 'statbox/common-config'),
)


KEYS_OWNER = 'STATINFRA'
ARC_TOKEN_NAME = 'ARC_TOKEN'


PY2_BINARY = 'python2'
PY3_BINARY = 'python3.6'
PY3_PACKAGE_PREFIX = 'python3-'


# STATLIBS_LXC_CONTAINER https://sandbox.yandex-team.ru/resource/2277776650/view
XENIAL_CONTAINER_RESOURCE = 2277776650
# STATLIBS_LXC_CONTAINER https://sandbox.yandex-team.ru/resource/2244271049
BIONIC_CONTAINER_RESOURCE = 2244271049


class StatlibsProjectBuilder(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        # resources
        disk_space = 30 * 1024
        ram = 16 * 1024
        cores = 8

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        priority = ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.NORMAL)
        kill_timeout = 3600
        max_restarts = 3

        # custom parameters
        with sdk2.parameters.Group('Build settings') as build_settings:
            with sdk2.parameters.String('Arcadia project', default=SUPPORTED_PROJECTS[0][0]) as arcadia_project:
                for name, path in SUPPORTED_PROJECTS:
                    arcadia_project.values[path] = name

            arcadia_revision = sdk2.parameters.String(
                'Arcadia revision', required=True
            )

    class Context(sdk2.Task.Context):
        binary_packages = None
        source_package = None
        build_version = None
        build_changes = None
        changelog_revision = None
        changelog_version = None
        tarball_resource = None

    arcadia_root = None
    # directory with setup.py
    project_dir = None
    # parent directory for debian directory
    debian_home = None

    def run_process(self, *args, **kws):
        return run_task_subprocess(self, *args, **kws)

    def on_execute(self):
        mount_arcadia_url = 'arcadia-arc:/#r{}'.format(self.Parameters.arcadia_revision)
        with arcadia_sdk.mount_arc_path(
            mount_arcadia_url,
            arc_oauth_token=sdk2.Vault.data(KEYS_OWNER, ARC_TOKEN_NAME),
        ) as arcadia_root:
            self.arcadia_root = arcadia_root

            self.collect_build_info()
            self.collect_package_info()
            self.do_build()
            self.do_postbuild()

    def do_build(self):
        pass

    def do_postbuild(self):
        pass

    def collect_build_info(self):
        self.project_dir = os.path.join(
            self.arcadia_root, self.Parameters.arcadia_project
        )

        logging.info('Getting build info...')
        info_file = os.path.abspath('build_info.json')
        self.run_process(
            [PY2_BINARY, 'setup.py', 'write_build_info', '--dst-file', info_file],
            cwd=self.project_dir,
            logger='write_build_info',
        )

        with open(info_file, 'rb') as source:
            build_info = json.load(source)
            self.debian_home = os.path.dirname(build_info['debian_directory'])
            package_root = build_info['package_root']

        logging.info('Debian home: %s', self.debian_home)
        logging.info('Package root: %s', package_root)

        if not os.path.isdir(self.debian_home):
            raise RuntimeError('parent directory for debian is missing')

        if not os.path.isdir(package_root):
            raise RuntimeError('package root directory is missing')

    def collect_package_info(self):
        arcadia_revision = self.Parameters.arcadia_revision
        changelog_file = os.path.join(self.debian_home, 'debian/changelog')

        logging.info('Extracting package names...')
        source_package, binary_packages = self._get_package_names()
        self.set_info('Source package name: {}'.format(source_package))
        self.Context.source_package = source_package
        self.Context.binary_packages = binary_packages

        logging.info('Extracting package version from changelog...')
        build_version = self._get_changelog_field('version')
        logging.info('Changelog version: %s', build_version)
        self.Context.changelog_version = build_version

        logging.info('Getting changelog revision...')
        arc_client = arc.Arc(arc_oauth_token=sdk2.Vault.data(KEYS_OWNER, ARC_TOKEN_NAME))
        changelog_revision = str(arc_client.log(
            self.arcadia_root,
            path=changelog_file,
            max_count=1,
            as_dict=True,
        )[0]['revision'])
        logging.info('Changelog revision: %s', changelog_revision)
        self.Context.changelog_revision = changelog_revision

        logging.info('Getting version to build...')
        # Generate changelog entry with develop version, if there are commits
        # after last release.
        if changelog_revision != arcadia_revision:
            # Here we use python compatible versioning. Our goal is to make
            # develop version greater than previous release version. So we use
            # post-release suffix with current svn revision (this also
            # guarantees right order for develop versions).
            # For more information see https://www.python.org/dev/peps/pep-0440/.
            build_version = '{}.post{}'.format(build_version, arcadia_revision)
            message = 'Test build for revision {}'.format(arcadia_revision)
            self._update_changelog(build_version, message)

        self.set_info('Package version: {}'.format(build_version))
        self.Context.build_version = build_version

        logging.info('Extracting build changes...')
        build_changes = self._get_changelog_field('changes').decode('utf8')
        self.set_info(build_changes)
        self.Context.build_changes = build_changes

        self.Context.save()

    def _get_package_names(self):
        source_package = None
        binary_packages = []

        def get_name(line):
            return line.partition(':')[-1].strip()

        with open(os.path.join(self.debian_home, 'debian/control')) as source:
            for line in source:
                if line.startswith('Source:'):
                    source_package = get_name(line)
                elif line.startswith('Package:'):
                    binary_packages.append(get_name(line))

        if not binary_packages or source_package is None:
            raise RuntimeError('Failed to parse control file')

        if len(binary_packages) > 2:
            raise RuntimeError(
                'Expect no more than two binary packages, got {}'
                .format(binary_packages)
            )

        py3_count = len([
            _ for _ in binary_packages
            if _.startswith(PY3_PACKAGE_PREFIX)
        ])

        if len(binary_packages) == 2 and py3_count != 1:
            raise RuntimeError(
                'In case of two binary packages expect exactly one of them '
                'to be python3-*, got {}'
                .format(binary_packages)
            )

        return source_package, binary_packages

    def _get_changelog_field(self, field):
        result = self.run_process(
            ['dpkg-parsechangelog', '-S', field],
            cwd=self.debian_home,
            read_stdout=True,
            logger='dpkg_parsechangelog_{}'.format(field),
        )

        return result.strip()

    def _update_changelog(self, version, message, distribution='xenial'):
        dch_cmd = [
            'dch',
            '--distribution={}'.format(distribution),
            '--newversion={}'.format(version),
            message
        ]

        logging.info('Updating changelog...')
        self.run_process(dch_cmd, cwd=self.debian_home, logger='dch')

        logging.info('Checking updated changelog...')
        if self._get_changelog_field('version') != version:
            raise RuntimeError('Failed to update changelog to given version')
