# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import os


class PypiConfig(object):
    def __init__(
        self,
        access_key,
        secret_key,
        index_server='yandex',
        repository='https://pypi.yandex-team.ru/simple/'
    ):
        self._access_key = access_key
        self._secret_key = secret_key
        self._index_server = index_server
        self._repository = repository
        self._config_path = None
        self._old_config_backup = None

    def __enter__(self):
        self._config_path = os.path.expanduser('~/.pypirc')

        if os.path.isfile(self._config_path):
            self._old_config_backup = '{}.backup'.format(self._config_path)
            os.rename(self._config_path, self._old_config_backup)

        self._write_config(self._config_path)

    def __exit__(self, *exc_info):
        os.remove(self._config_path)

        if self._old_config_backup is not None:
            os.rename(self._old_config_backup, self._config_path)

        self._config_path = None
        self._old_config_backup = None

    @property
    def index_server(self):
        return self._index_server

    @property
    def repository(self):
        return self._repository

    @property
    def path(self):
        return self._config_path

    def _write_config(self, path):
        content = (
            '[distutils]\n'
            'index-servers =\n'
            '    {index_server}\n'
            '\n'
            '[{index_server}]\n'
            'repository: {repository}\n'
            'username: {access_key}\n'
            'password: {secret_key}'
        ).format(
            index_server=self._index_server,
            repository=self._repository,
            access_key=self._access_key,
            secret_key=self._secret_key,
        )

        with open(path, 'wb') as sink:
            sink.write(content)
