# -*- coding: utf-8 -*-

import glob
import logging
import os
import tarfile


from sandbox import sdk2
from sandbox.projects.common import link_builder as lb

from ..utils.project_builder import (
    StatlibsProjectBuilder, PY2_BINARY,
    BIONIC_CONTAINER_RESOURCE,
)
from ..utils.pypi import PypiConfig
from ..utils.resources import STATLIBS_WHEELS_TARBALL


DEFAULT_PY3_VERSION = '3.6'
DEFAULT_PY2_VERSION = '2.7'

ALLOWED_PY3_VERSIONS = ('3.5', '3.6', '3.7', '3.8', '3.9')

PYPI_REPOS = (
    ('Default', 'https://pypi.yandex-team.ru/simple/'),
    ('Arcadia', 'https://pypi.yandex-team.ru/arcadia/'),
)


class Platforms(object):
    LINUX = 'linux'
    MACOSX = 'macosx'
    WINDOWS = 'windows'
    ALL = 'all'


class StatlibsWheels(StatlibsProjectBuilder):
    """Build & upload wheels for one of statbox projects in Arcadia"""

    class Requirements(StatlibsProjectBuilder.Requirements):
        container_resource = BIONIC_CONTAINER_RESOURCE

    class Parameters(StatlibsProjectBuilder.Parameters):
        with sdk2.parameters.Group('Release settings') as release_settings:
            universal_build = sdk2.parameters.Bool(
                'Is pure python package', default=False
            )
            with universal_build.value[False]:
                with sdk2.parameters.CheckGroup('Wheels platform(s)') as platforms:
                    platforms.values[Platforms.LINUX] = platforms.Value('Linux', checked=True)
                    platforms.values[Platforms.MACOSX] = platforms.Value('MacOS X', checked=True)
                    platforms.values[Platforms.WINDOWS] = platforms.Value('Windows', checked=True)

            build_py2 = sdk2.parameters.Bool('Build wheels for Python2', default=True)
            build_py3 = sdk2.parameters.Bool('Build wheels for Python3', default=False)
            with build_py3.value[True], universal_build.value[False]:
                with sdk2.parameters.CheckGroup('Python3 version(s)') as py3_versions:
                    for version in ALLOWED_PY3_VERSIONS:
                        py3_versions.values[version] = py3_versions.Value(version, checked=True)

            upload_wheels = sdk2.parameters.Bool('Upload wheels to PyPI', default=False)
            with upload_wheels.value[True]:
                with sdk2.parameters.String('PyPI repository', default=PYPI_REPOS[0][1]) as pypi_repository:
                    for name, url in PYPI_REPOS:
                        pypi_repository.values[url] = name

    # directory with wheels
    wheels_dir = None

    def do_build(self):
        logging.info('Start build actions')

        need_upload = self.Parameters.upload_wheels

        if need_upload:
            if self.Parameters.arcadia_revision != self.Context.changelog_revision:
                raise RuntimeError(
                    'Uploading allowed only for release revisions'
                )

        python_versions = self._get_python_versions()
        logging.debug('Python versions: %s', python_versions)

        platforms = self._get_platforms()
        logging.debug('Platforms: %s', platforms)

        self.wheels_dir = str(self.path('WHEELS').absolute())
        logging.debug('Wheels directory: %s', self.wheels_dir)

        for platform in platforms:
            for python_version in python_versions:
                self._build_wheels(platform, python_version)

                logging.info('Cleaning after wheels build...')
                self.run_process(
                    [PY2_BINARY, 'setup.py', 'clean', '--all'],
                    cwd=self.project_dir,
                    logger='setup_clean_{}'.format(platform),
                )

        if need_upload:
            self._upload_wheels()

    def do_postbuild(self):
        self.make_resource()

    def make_resource(self):
        package = self.Context.source_package
        version = self.Context.build_version

        tarball_path = os.path.abspath(
            '{}_{}_wheels.tar.gz'.format(package, version)
        )

        logging.info('Creating wheels for %s=%s...', package, version)
        logging.debug('Tarball path: %s', tarball_path)

        wheels_pattern = os.path.join(self.wheels_dir, '*.whl')
        logging.debug('Wheels pattern: %s', wheels_pattern)

        wheels_mapping = {
            _: os.path.basename(_) for _ in glob.glob(wheels_pattern)
        }
        logging.debug('Wheel files: %s', wheels_mapping.keys())
        self.set_info('Wheels: {}'.format(wheels_mapping.values()))

        with tarfile.open(tarball_path, mode='w:gz') as tarball:
            for path, filename in wheels_mapping.iteritems():
                tarball.add(path, arcname=filename)

        logging.debug('Tarball size: %s', os.stat(tarball_path).st_size)

        tarball_resource = STATLIBS_WHEELS_TARBALL(
            self,
            description='Wheels for {}={}'.format(package, version),
            path=tarball_path,
            # attributes
            ttl=30,
            package=package,
            version=version,
            platforms=';'.join(self._get_platforms()),
            pyversions=';'.join(self._get_python_versions()),
        )

        sdk2.ResourceData(tarball_resource).ready()

        self.Context.tarball_resource = tarball_resource.id

        html_link = lb.resource_link(tarball_resource.id)

        self.set_info('Wheels tarball: {}'.format(html_link), do_escape=False)

    def _get_platforms(self):
        if self.Parameters.universal_build:
            return [Platforms.ALL]

        return self.Parameters.platforms

    def _get_python_versions(self):
        versions = []

        if self.Parameters.build_py2:
            versions.append(DEFAULT_PY2_VERSION)

        if self.Parameters.build_py3:
            if self.Parameters.universal_build:
                versions.append(DEFAULT_PY3_VERSION)
            else:
                versions.extend(self.Parameters.py3_versions)

        return versions

    @staticmethod
    def _prepare_build_environment(python_version):
        env = os.environ.copy()
        env.update(STATLIBS_PYTHON_VERSION=python_version)
        return env

    def _build_wheels(self, platform, python_version):
        logging.info('Building wheels for Python%s %s...', python_version, platform)

        build_cmd = [
            PY2_BINARY,
            'setup.py',
            'bdist_wheel',
            '-d', self.wheels_dir,
            '--platform', platform,
        ]

        self.run_process(
            build_cmd,
            cwd=self.project_dir,
            env=self._prepare_build_environment(python_version),
            logger='setup_bdist_wheel_{}'.format(platform),
        )

        self.set_info(
            'Wheels for Python{} {} was successfully built'.format(python_version, platform)
        )

    def _upload_wheels(self):
        pypi_repo = self.Parameters.pypi_repository

        pypi_config = PypiConfig(
            access_key=sdk2.Vault.data('STATINFRA', 'robot-statinfra-pypi-access'),
            secret_key=sdk2.Vault.data('STATINFRA', 'robot-statinfra-pypi-secret'),
            repository=pypi_repo,
        )

        wheels = os.path.join(self.wheels_dir, '*.whl')

        with pypi_config:
            upload_cmd = [
                'twine',
                'upload',
                wheels,
                '-r', pypi_config.index_server,
                '--config-file', pypi_config.path,
                '--disable-progress-bar',
                '--skip-existing'
            ]

            self.run_process(
                upload_cmd, logger='upload_wheels'
            )

        self.set_info(
            'All wheels were successfully uploaded to {}'.format(pypi_repo)
        )
