# coding: utf-8
import os
import json
import logging
import tempfile

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.stop_leak.common.yahec import HecSender
from sandbox.projects.stop_leak.common import sectools
from sandbox.projects.stop_leak.common import http
from sandbox.projects.stop_leak.resource_types import AntSecretResults
import sandbox.sandboxsdk.svn as svn
import sandbox.sandboxsdk.ssh as ssh


CHECKOUT_URL = 'svn+ssh://teamcity@arcadia.yandex.ru/adm/trunk'
VCS = 'arcadm'


class AuditArcAdmSec(sdk2.Task):
    ant_secret_session = None

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        excludes = sdk2.parameters.String('Excludes', default='.git,.hg,.svn,node_modules,.venv,ut')
        store_results = sdk2.parameters.Bool('Send results to Splunk', default=True)
        compromized = sdk2.parameters.Bool('Compromise secrets', default=False)
        debug_mode = sdk2.parameters.Bool('Enable debug mode', default=False)
        not_ignore = sdk2.parameters.Bool('Ignore ".secretsignore" files', default=False)
        save_to_res = sdk2.parameters.Bool('Save ant-secret results in resources', default=False)
        valid_only = sdk2.parameters.Bool('Report only validated secrets', default=True)

    def compromise(self, secret):
        self.ant_secret_session.post(
            'https://ant.sec.yandex-team.ru/api/v1/touch-known',
            json={
                'secret': secret,
                'meta': {
                    'vcs': VCS,
                    'task_id': self.id,
                }
            }
        )

    def on_execute(self):
        ant_path = sectools.download_secret_search()

        cmd = [ant_path, '--status-code=0', '--format=jsonlines']
        if self.Parameters.valid_only:
            cmd.append('--valid-only')
        else:
            cmd.append('--validate')
        if self.Parameters.excludes:
            cmd.append('--excludes=' + self.Parameters.excludes)
        if self.Parameters.not_ignore:
            cmd.append('--not-ignore')

        svn_root = tempfile.mkdtemp('arcadia')
        cmd.append(svn_root)

        with ssh.Key(self, self.owner, 'STOP_LEAK_SSH_KEY'):
            checkout(CHECKOUT_URL, svn_root)
            env = os.environ.copy()
            env['ANT_INTERNAL_TOKEN'] = sdk2.Vault.data('STOP_LEAK_ANT_INTERNAL_TOKEN')

            if self.Parameters.debug_mode:
                self.suspend()

            ant_out_path = str(self.path('ant-secret.json'))
            with open(ant_out_path, 'wt') as fd:
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('ant-secret')) as pl:
                    sp.check_call(cmd, env=env, stdout=fd, stderr=pl.stdout)

            if self.Parameters.save_to_res:
                ant_resource = sdk2.ResourceData(
                    AntSecretResults(
                        self,
                        'Ant-Secret search results',
                        ant_out_path
                    )
                )
                ant_resource.ready()

            if not self.Parameters.store_results:
                return

            mark_compromized = self.Parameters.compromized
            if mark_compromized:
                self.ant_secret_session = http.requests_retry_session()
                self.ant_secret_session.headers.update({'X-Internal-Token': sdk2.Vault.data('STOP_LEAK_ANT_INTERNAL_TOKEN')})

            with HecSender(sdk2.Vault.data('STOP_LEAK_HEC_TOKEN')) as log:
                with open(ant_out_path, 'rt') as results_fd:
                    for line in results_fd:
                        line = line.strip()
                        if not line:
                            continue

                        result = json.loads(line)
                        info = get_svn_info(result['path'])
                        for value in result.get('secrets', []):
                            line_no = value.get('line_no', 0)
                            if line_no > 0 and 'blame' in info:
                                author = info['blame'][line_no - 1]
                            else:
                                author = info['author']

                            additional = value.get('additional', {})
                            sha1 = ''
                            if 'sha1' in additional:
                                sha1 = additional['sha1']
                                del additional['sha1']

                            # backward compatibility
                            if 'type' in value:
                                additional['secret_validator'] = value['type']

                            report = dict(
                                sha1=sha1,
                                url='{url}?rev={rev}#L{lo}'.format(url=info['url'], rev=info['revision'], lo=line_no),
                                private=False,
                                ignore=value.get('ignore', False),
                                vcs=VCS,
                                user=author,
                                state_id=0,
                                type=value.get('type', None),
                                secret=value.get('secret', None),
                                additional=additional,
                                holders=None,
                                secret_validated=value.get('validated', False)
                            )
                            log.send(**report)

                            if mark_compromized:
                                secret = value.get('secret', None)
                                # TODO(buglloc): ugly!
                                if not secret:
                                    secret = additional.get('ssh_fingerprint', None)
                                if not secret:
                                    logging.info("can't find secret in ant-secret report")
                                else:
                                    self.compromise(secret)


def get_svn_info(path):
    info = svn.Svn.info(path)
    blame = svn.Svn.blame(path).strip()
    result = {
        'url': prepare_url(info.get('url', '')),
        'author': info.get('author', 'unknown'),
        'revision': info.get('entry_revision', 0)
    }
    if blame:
        result['blame'] = map(lambda x: x.split()[1], blame.split('\n'))
    return result


def prepare_url(url):
    return url \
        .replace(CHECKOUT_URL, 'https://arc.yandex-team.ru/wsvn/adm/trunk')


def checkout(url, path):
    opts = svn.Svn.SvnOptions()
    opts.force = True
    # SB may fail to checkout svn externals
    opts.ignore_externals = True
    r = svn.Svn.svn('checkout', opts=opts, url=url, path=path, summary=True)
    logging.info('svn checkout summary:\n%s', r.stdout.strip())
