# coding: utf-8
import json
import tempfile

from sandbox import sdk2
import sandbox.common.types.misc as ctm
from sandbox.projects.stop_leak.base_hector_task.hector_git import BaseHectorGitTask


class AuditExternalGithubSec(BaseHectorGitTask):
    vcs = 'github_external'
    hector_command = 'gh'
    use_ssh = False

    class Requirements(BaseHectorGitTask.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(BaseHectorGitTask.Parameters):
        orgs = sdk2.parameters.String('List of organization', default='', multiline=True)
        staff = sdk2.parameters.Bool('Check staff-user repos', default=True)
        search_query = sdk2.parameters.String('Scan all repos by search query', default='')

    def _hector_env(self):
        env = super(AuditExternalGithubSec, self)._hector_env()
        env['GH_TOKEN'] = sdk2.Vault.data('STOP_LEAK_GITHUB_EXERNAL_TOKEN')
        return env

    def _hector_command(self):
        hector_cmd, ant_secret_cmd = super(AuditExternalGithubSec, self)._hector_command()
        hector_cmd.append('--gh-url=https://api.github.com/')
        hector_cmd.append('--no-forks')
        hector_cmd.append('--search-query=%s' % self.Parameters.search_query)
        hector_cmd.append('--config=%s' % self.prepare_config())
        return hector_cmd, ant_secret_cmd

    def _parse_report(self, report, repo):
        report = super(AuditExternalGithubSec, self)._parse_report(report, repo)
        # FIXME!
        found = False
        if self.Parameters.staff:
            for login, r in self.staff_github_repos:
                url = report.get('url')
                if not url:
                    continue
                if url.lower().startswith('https://github.com/%s' % r):
                    found = True
                    report['user'] = login
                    report['holders'] = [login]
                    break
        if not found:
            report['user'] = 'unknown'
            report['holders'] = ['unknown']
        return report

    def prepare_config(self):
        orgs = []
        if self.Parameters.orgs.strip():
            for r in self.Parameters.orgs.strip().split('\n'):
                if '#' not in r:
                    orgs.append(r.strip())

        users = []
        if self.Parameters.staff:
            for _, r in self.staff_github_repos:
                users.append(r)
        return self.save_repos(orgs, users)

    def save_repos(self, orgs, users):
        config = {
            'Projects': orgs,
            'Users': users,
        }

        cfg_path = tempfile.NamedTemporaryFile(suffix='.hector.json').name
        with open(cfg_path, 'wt') as f:
            f.write(json.dumps(config))
        return cfg_path
