import socket
from socket import error as SocketError
import logging
import json
import time


class Facility:
    "Syslog facilities"
    KERN, USER, MAIL, DAEMON, AUTH, SYSLOG, \
        LPR, NEWS, UUCP, CRON, AUTHPRIV, FTP = range(12)

    LOCAL0, LOCAL1, LOCAL2, LOCAL3, \
        LOCAL4, LOCAL5, LOCAL6, LOCAL7 = range(16, 24)


class Level:
    "Syslog levels"
    EMERG, ALERT, CRIT, ERR, \
        WARNING, NOTICE, INFO, DEBUG = range(8)


class YaSyslog:
    """
    A syslog client that logs to a remote server.
    """

    def __init__(self, name, host='trapdoor.yandex.net', port=514, facility=Facility.USER):
        self.host = host
        self.port = port
        self.facility = facility
        self.name = name
        self.socket = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)

    def send(self, message, level):
        "Send a syslog message to remote host using UDP."
        data = '<{priority}>{progname}: {message}\n'.format(
            priority=(level + self.facility * 8),
            progname=self.name,
            message=message
        )

        send_try = 0
        first_try = True
        while True:
            try:
                if not first_try:
                    self.disconnect()
                    self.connect()
                self.socket.sendall(data)
                break
            except SocketError as e:
                first_try = False
                logging.warn('failed to send data to syslog', e)
                if send_try >= 6:
                    raise e
                send_try += 1
                # Try to reconnect and send one more time
                time.sleep(1 << send_try)

    def warn(self, message):
        "Send a syslog warning message."
        self.send(message, Level.WARNING)

    def notice(self, message):
        "Send a syslog notice message."
        self.send(message, Level.NOTICE)

    def error(self, message):
        "Send a syslog error message."
        self.send(message, Level.ERR)

    def connect(self):
        self.socket = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        self.socket.connect((self.host, self.port))

    def splunk(self, **kwargs):
        return self.warn(self.splunk_message(**kwargs))

    def splunk_message(self, **kwargs):
        additional = None
        if 'additional' in kwargs:
            additional = kwargs['additional']
            del kwargs['additional']
        result = format_message(kwargs.items())
        if additional:
            result += ' ' + format_message(additional.items())
        return result

    def disconnect(self):
        self.socket.close()

    def __enter__(self):
        self.connect()
        return self

    def __exit__(self, exception_type, exception_value, traceback):
        try:
            self.disconnect()
        except Exception:
            pass


def conv_value(value):
    if isinstance(value, list):
        value = ','.join(value)
    elif isinstance(value, dict):
        value = ' '.join('%s=%s' % (k, conv_inner_value(v)) for k, v in value.items())

    if value and isinstance(value, basestring):
        # Splunk doesn't like double quotes ^_^
        value = value.replace('"', "'").encode('utf-8')
    return json.dumps(value)


def conv_inner_value(value):
    if value:
        if isinstance(value, basestring):
            # Splunk doesn't like double quotes ^_^
            value = value.replace('"', "'").encode('utf-8')
            value = repr(value)
            if value[0] == '"':
                # Ugly hack
                value = "'%s'" % value[1:-1].replace('\'', '\\\'')
            return value
        elif isinstance(value, list):
            value = ','.join([conv_inner_value(v) for v in value])
        elif isinstance(value, dict):
            value = ' '.join('%s=%s' % (k, conv_inner_value(v)) for k, v in value.items())

    return json.dumps(value)


def format_message(items):
    return ' '.join('%s=%s' % (k, conv_value(v)) for k, v in items)
