import logging
import os

import sandbox.sdk2 as sdk2
from sandbox.projects.suggest.resource_types import SuggestEcomBuilderExecutable
from sandbox.sandboxsdk.process import run_process


class BuildEcomSuggestDict(sdk2.Task):
    ''' Build multi ecom dict '''

    class Parameters(sdk2.Task.Parameters):
        dict_prefix = sdk2.parameters.String('Dicts tables prefix', required=True)
        folder_name = sdk2.parameters.String('Dicts folder name', required=True)
        threads_count = sdk2.parameters.Integer('Threads count', default=32, required=True)
        thread_buffer_size = sdk2.parameters.Integer('Thread buffer size (in mb)', default=100, required=True)

        builder_owner = sdk2.parameters.String('SUGGEST_ECOM_BUILDER_EXECUTABLE owner', default='SUGGEST-TEST', required=True)
        vault_user = sdk2.parameters.String('Sandbox vault user with YT token', default='SUGGEST', required=True)
        vault_yt_token_name = sdk2.parameters.String('YT token name from Sandbox vault', default='SUGGEST_YT_TOKEN', required=True)
        yt_cluster = sdk2.parameters.String('YT cluster', default='hahn', required=True)
        resource_type = sdk2.parameters.String('Resource type', default='SUGGEST_ECOM_DICTS', required=True)
        resource_description = sdk2.parameters.String('Resource description', default='Ecom dicts', required=True)
        attributes = sdk2.parameters.Dict('Resource attributes', default={})

    def on_execute(self):
        # Write YT token from Vault to file
        token_value = sdk2.Vault.data(self.Parameters.vault_user, self.Parameters.vault_yt_token_name)
        with open('yt_token.txt', 'w') as t:
            t.write(token_value)

        export_dir = self.Parameters.folder_name
        os.mkdir(export_dir)

        # Get ecom builder executable
        ecom_builder = sdk2.Resource.find(resource_type=SuggestEcomBuilderExecutable, state='READY', owner=self.Parameters.builder_owner).first()
        ecom_builder_path = str(sdk2.ResourceData(ecom_builder).path)
        logging.info('ecom_builder_path: ' + ecom_builder_path)

        # Form builder command
        cmd = [ecom_builder_path, 'dicts-downloader',
               '--dicts-tables-prefix', self.Parameters.dict_prefix,
               '--output-folder', export_dir,
               '--cluster', self.Parameters.yt_cluster,
               '--yt-token-path', 'yt_token.txt',
               '--threads-count', str(self.Parameters.threads_count),
               '--thread-buffer-size', str(self.Parameters.thread_buffer_size)]
        logging.info('Run command: ' + ' '.join(cmd))

        # Run builder command
        run_process(cmd, log_prefix="dicts_downloader", wait=True, shell=True)
        logging.info('ecom dicts have been downloaded from yt.')

        # Publish rich storage dict
        resource_type = sdk2.Resource[self.Parameters.resource_type]
        dict_resource = resource_type(self, self.Parameters.resource_description, export_dir, **self.Parameters.attributes)
        sdk2.ResourceData(dict_resource).ready()
