import logging
import os

import sandbox.sdk2 as sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.suggest.resource_types import RichStorageBuilderExecutable


class BuildRichStorage(sdk2.Task):
    ''' Build rich storage dict '''

    class Parameters(sdk2.Task.Parameters):
        export_table = sdk2.parameters.String('Export table path', required=True)
        rows_limit = sdk2.parameters.Integer('Rows limit')
        check_sum_bits = sdk2.parameters.Integer('Bits count for check sums', default=32)
        verbose = sdk2.parameters.Bool('Verbose build', default=False)
        external_facts = sdk2.parameters.Bool('External facts', default=False)
        skip_factors = sdk2.parameters.Bool('Skip factors', default=False)
        builder_owner = sdk2.parameters.String('RICH_STORAGE_BUILDER_EXECUTABLE owner', default='SUGGEST-TEST', required=True)
        vault_user = sdk2.parameters.String('Sandbox vault user with YT token', default='SUGGEST', required=True)
        vault_yt_token_name = sdk2.parameters.String('YT token name from Sandbox vault', default='SUGGEST_YT_TOKEN', required=True)
        yt_cluster = sdk2.parameters.String('YT cluster', default='hahn', required=True)
        resource_type = sdk2.parameters.String('Resource type', default='SUGGEST_RICH_STORAGE', required=True)
        resource_description = sdk2.parameters.String('Resource description', default='Rich storage', required=True)
        attributes = sdk2.parameters.Dict('Resource attributes', default={})

    def on_execute(self):
        # Write YT token from Vault to file
        token_value = sdk2.Vault.data(self.Parameters.vault_user, self.Parameters.vault_yt_token_name)
        with open('yt_token.txt', 'w') as t:
            t.write(token_value)

        export_dir = 'dict'
        os.mkdir(export_dir)

        # Get rich storage builder
        rich_storage_builder = sdk2.Resource.find(resource_type=RichStorageBuilderExecutable, state='READY', owner=self.Parameters.builder_owner).first()
        rich_storage_builder_path = str(sdk2.ResourceData(rich_storage_builder).path)
        logging.info('rich_storage_builder_path: ' + rich_storage_builder_path)

        # Form builder command
        cmd = [rich_storage_builder_path,
               '--export-table', self.Parameters.export_table,
               '--prefix', export_dir + '/dict',
               '--cluster', self.Parameters.yt_cluster,
               '--token',  'yt_token.txt']
        if self.Parameters.rows_limit:
            cmd.append('--rows-limit')
            cmd.append(str(self.Parameters.rows_limit))
        if self.Parameters.check_sum_bits:
            cmd.append('--check-sum-bits')
            cmd.append(str(self.Parameters.check_sum_bits))
        if self.Parameters.verbose:
            cmd.append('--verbose')
        if self.Parameters.external_facts:
            cmd.append('--external-facts')
        if self.Parameters.skip_factors:
            cmd.append('--skip-factors')
        logging.info('Run command: ' + ' '.join(cmd))

        # Run builder command
        with sdk2.helpers.ProcessLog(self, logger='rich_storage_builder') as pl:
            subprocess.check_call(cmd, stdout=pl.stdout, stderr=subprocess.STDOUT)
        logging.info('rich_storage has builded.')

        # Publish rich storage dict
        resource_type = sdk2.Resource[self.Parameters.resource_type]
        dict_resource = resource_type(self, self.Parameters.resource_description, export_dir, **self.Parameters.attributes)
        sdk2.ResourceData(dict_resource).ready()
