import logging
import os

from sandbox import sdk2

import sandbox.projects.mediasearch.resource_types as rt
import sandbox.projects.suggest.dicts as suggest_dicts
import sandbox.projects.common.file_utils as fu
import sandbox.projects.common.ya_deploy.release_integration as ri
import sandbox.sandboxsdk.environments as envs


class BuildKinopoiskSuggestDict(sdk2.Task, suggest_dicts.SuggestDictTask, ri.ReleaseToYaDeployTask2):

    class Parameters(sdk2.Parameters):

        yt_token_secret_id = sdk2.parameters.String('yav YT token secret ID', required=True)
        yt_token_version_id = sdk2.parameters.String('yav YT token version ID', required=True)
        yt_path = sdk2.parameters.String('Input table path', required=True)

        yp_token_yav_secret_id = sdk2.parameters.String('yav YP token secret ID', required=False)

    class Requirements(sdk2.Requirements):

        environments = [
            envs.PipEnvironment('yandex-yt'),
            envs.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    def on_execute(self):
        import yt.wrapper as yt
        yt_token = sdk2.yav.Secret(
            secret=self.Parameters.yt_token_secret_id,
            version=self.Parameters.yt_token_version_id,
        ).data()['secret']
        yt_client = yt.YtClient(proxy='hahn.yt.yandex.net', token=yt_token)
        queries = []
        data = []
        logging.debug('Reading data from %s', self.Parameters.yt_path)
        total_frequency = 0
        for rec in yt_client.read_table(self.Parameters.yt_path):
            query = rec['title']
            queries.append('{}\t\t{}\t\n'.format(
                query,
                str(rec['frequency']),
            ))
            data.append('{}\t{}\n'.format(
                query,
                str(rec['data']),
            ))
            total_frequency += rec['frequency']

        logging.debug('Saving data to files')
        cwd = os.getcwd()
        queries_path = os.path.join(cwd, 'queries')
        fu.write_lines(queries_path, sorted(queries))
        data_path = os.path.join(cwd, 'data')
        fu.write_lines(data_path, data)
        groups_path = os.path.join(cwd, 'groups')
        fu.write_lines(groups_path, ['\n'])
        streams_path = os.path.join(cwd, 'streams')
        fu.write_lines(streams_path, ['ALL\t{}\n'.format(str(total_frequency))])

        dict_path = os.path.join(cwd, 'dict')
        os.makedirs(dict_path)
        dict_prefix = os.path.join(dict_path, 'dict')
        logging.debug('Building dictionaries')
        self.run_data_builder(
            dict_prefix,
            queries_path,
            groups_path,
            streams_path,
            data_path,
            word_index=True,
        )

        resource = rt.KinopoiskLingboostSuggestDict(
            self,
            'Kinopoisk suggest lingboost dictionary',
            path=dict_path,
            name='dict',
            autodeploy='yes',
        )
        sdk2.ResourceData(resource).ready()

    def on_release(self, parameters_):
        if self.Parameters.yp_token_yav_secret_id is not None:
            ri.ReleaseToYaDeployTask2.on_release(self, parameters_)
        else:
            logging.debug('No YP token provided for this task, skipping release to y.deploy')
        self.mark_released_resources(parameters_["release_status"], ttl=7)
