import os
import shutil
import tarfile

from sandbox import sdk2
from sandbox.projects.common import utils
from sandbox.projects.resource_types import NEWS_BACKOFFICE_DATA, SUGGEST_DICT, ARCADIA_PROJECT
from sandbox.sandboxsdk.process import run_process


DICT_NAME = 'news.media.params2'


class SuggestDictTask:
    @staticmethod
    def get_data_builder():
        return utils.sync_last_stable_resource('SUGGEST_DATA_BUILDER_EXECUTABLE')

    def run_data_builder(
        self,
        dict_prefix,
        queries_path,
        groups_path,
        streams_path,
        data_path='',
        word_index=False,
        top_size=10,
        thread_count=8,
        refs_path='',
        export_table='',
        stop_word_count=300000,
        stop_prefix_count=150000,
        streams_without_subtract_path='',
    ):
        data_builder_path = self.get_data_builder()
        command = [
            data_builder_path,
            '-o', dict_prefix,
            '-r', queries_path,
            '-g', groups_path,
            '-s', streams_path,
            '-c', top_size,
            '-T', thread_count,
            '-R', 0,
            '--stop-word-count', stop_word_count,
            '--stop-prefix-count', stop_prefix_count,
        ]

        if data_path:
            command.extend(['-d', data_path])

        if word_index:
            command.extend(['-A'])

        if refs_path:
            command.extend(['-f', refs_path])

        if export_table:
            command.extend(['--export-table', export_table])

        if streams_without_subtract_path:
            command.extend(['--streams-without-subtract', streams_without_subtract_path])

        run_process(command, log_prefix='bin_data_builder', wait=True, shell=True)

    def publish_dict(self, name, description, path, autodeploy=False, ttl='inf'):
        params = {'ttl': ttl}
        if autodeploy:
            params['autodeploy'] = 'yes'

        dict_resource = SUGGEST_DICT(self, description, path=path, name=name, **params)
        sdk2.ResourceData(dict_resource).ready()


class BuildNewsSuggestDict(sdk2.Task, SuggestDictTask):
    """ Build news suggest dictionary """

    class Parameters(sdk2.Task.Parameters):
        binary = sdk2.parameters.Resource(
            'Executable',
            required=True,
            resource_type=ARCADIA_PROJECT,
            default=2046777118,
        )

        deploy = sdk2.parameters.Bool(
            'Deploy',
            required=True,
            default=False,
        )

    class Requirements(sdk2.Requirements):
        pass

    def on_execute(self):
        news_data_path = self.get_news_data()
        tarfile.open(str(news_data_path)).extractall()

        queries = 'queries'
        groups = 'groups'
        streams = 'streams'
        data_answers = 'data'
        data_meta = 'data_params'
        dict_path = DICT_NAME
        os.makedirs(dict_path)
        dict_prefix = os.path.join(dict_path, 'bindata')

        run_process(
            [sdk2.ResourceData(self.Parameters.binary).path / 'bin' / 'bin'],
            log_prefix='data_builder',
            wait=True,
            shell=True,
        )

        shutil.copy(data_answers, dict_prefix + '.nav_answers')
        shutil.copy(data_meta, dict_prefix + '.nav_params.answers')

        self.run_data_builder(
            dict_prefix,
            queries,
            groups,
            streams,
            top_size=100
        )

        self.publish_dict(
            DICT_NAME,
            'Dictionaries for News suggest',
            dict_path,
            autodeploy=self.Parameters.deploy,
        )

    @staticmethod
    def get_news_data():
        return utils.sync_last_stable_resource(NEWS_BACKOFFICE_DATA)
