import os
import os.path
import logging
import shutil

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk.process import run_process
from sandbox import common
from sandbox.projects.suggest import resource_types
from sandbox.projects.common.ui import build_ui
from sandbox.projects.common.vcs.arc import Arc

from cryptography.fernet import Fernet

class ArcadiaConsts(object):
    SANDBOX_TAGS_URL = "arcadia:/arc/tags/sandbox"
    SANDBOX_BRANCHES_URL = "arcadia:/arc/branches/sandbox"
    SANDBOX_TRUNK_URL = "arcadia:/arc/trunk/arcadia/sandbox"
    ARCADIA_YA_PATH = "arcadia:arc/trunk/arcadia/ya"


# ------------------------------------------------------------------------------------------------------------------
def checkout_from_arc(desc, path, yav_token_secret_id):
    res = str(sdk2.task.Task.current.path(desc))

    logging.info('getting token for yav')
    yav_token_secret = sdk2.yav.Secret(yav_token_secret_id)
    yav_token = yav_token_secret.data()['yav-token']
    logging.info("Got yav token *" + yav_token[-3:])

    logging.info('mounting with arc tool')
    arc = Arc(yav_oauth_token=yav_token)
    with arc.mount_path('', 'trunk', fetch_all=False, minimize_mount_path=False) as trunk_path:
        target_path = os.path.join(trunk_path, path)
        logging.info('coping {} to {}'.format(target_path, res))
        shutil.copytree(target_path, res)

    cwd_files = ' '.join(os.listdir(res))
    logging.info('files: %s', cwd_files)

    return res


# end checkout_from_arc
# ------------------------------------------------------------------------------------------------------------------


class SuggestBuildFiltersArcDebug(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("properties") as props:
            autodeploy = sdk2.parameters.String(
                "autodeploy dict yes|no",
                name="autodeploy",
                default='no',
                required=False
            )
            yav_token_secret_id = sdk2.parameters.String(
                "yav token secret id in yav (it should have the key \"yav-token\")",
                name="yav_token_secret_id",
                default="sec-01fxsqtz6j9czny1kyvfah90df",
                required=False
            )
            fernet_key_secret_id = sdk2.parameters.String(
                "fernet key secret id in yav (it should have the key \"fernet-key\")",
                name="fernet_key_secret_id",
                default="sec-01fxsggw5tp12qdxh2zb3n4g9a",
                required=False
            )

    def _clone_arc(self, inf):
        basepath = str(self.path("arc"))
        if os.path.isdir(basepath):
            # Arc is already cloned
            return basepath

        sdk2.svn.Arcadia.export(ArcadiaConsts.ARCADIA_YA_PATH, str(self.path()))
        with sdk2.helpers.ProcessLog(self, logger="ya_clone") as pl:
            process = sp.Popen(
                [
                    str(self.path("ya")), "clone",
                    "--revision={}".format(inf.rev),
                    "--branch={}".format(inf.branch),
                    basepath,
                ],
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            )
            process.wait()
            if process.returncode:
                raise common.errors.SubprocessError(
                    "process {0} died with exit code {1}".format(process.pid, process.returncode)
                )
        return basepath

    def _ci_from_branch(self):
        branch = 'trunk'  # self.Parameters.sandbox_branch_name
        if not branch:
            raise common.errors.TaskFailure("Sandbox branch name is not specified. Cannot execute the task.")
        branch, rev = (branch.split("@", 1) + [None])[:2]
        if branch == "trunk":
            branch_url = ArcadiaConsts.SANDBOX_TRUNK_URL
        else:
            branch_url = "{}/{}".format(ArcadiaConsts.SANDBOX_BRANCHES_URL, branch)
        if not rev:
            rev = sdk2.svn.Arcadia.get_revision(branch_url)
        return build_ui.CodeInfoSvn(None, branch, rev, branch_url)

    def on_execute(self):
        platform = 'linux'
        inf = self._ci_from_branch()
        arc = self._clone_arc(inf)
        logging.info('arc=%s', arc)

        fernet_key_secret = sdk2.yav.Secret(self.Parameters.fernet_key_secret_id)
        fernet_key = fernet_key_secret.data()['fernet-key']

        logging.info('checkout suggest_dict from svn')
        path_suggest_dict = checkout_from_arc('suggest_dict', 'quality/trailer/suggest_dict', self.Parameters.yav_token_secret_id)

        logging.info('BUILD BINARIES')
        os.mkdir('./bin')
        binaries = [('quality/trailer/suggest_dict/tools/forms_generator', 'generator'),
                    ('quality/trailer/suggest_dict/tools/regexp_list_precom', 'regexp_list_precom'),
                    ('dict/gazetteer/compiler', 'gztcompiler')]

        for (path, name) in binaries:
            logging.info('start build %s', path)
            with sdk2.helpers.ProcessLog(self,
                                         logger="ya_make_{}_{}".format(os.path.basename(path), platform)) as pl:
                process = sp.Popen(
                    [
                        os.path.join(arc, "ya"), "make", "--checkout", "--thin", "--build=release",
                        "--target-platform={}".format(platform),
                        "-o", '.',
                        path,
                    ],
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                    cwd=arc
                )
                process.wait()
                if process.returncode:
                    raise common.errors.SubprocessError(
                        "process {0} died with exit code {1}".format(process.pid, process.returncode)
                    )
                logging.info('done build %s', name)
            os.symlink(arc + '/' + path + '/' + name, './bin/' + name)

        logging.info('BUILD gzt files')

        os.mkdir('./gzt')
        path = 'quality/trailer/suggest_dict'

        logging.info('prepare')
        with sdk2.helpers.ProcessLog(self, logger='prepare') as pl:
            cmd = path_suggest_dict + '/tools/prepare_data.py' \
                  + ' --suggest-dict-path ' + path_suggest_dict \
                  + ' -b ./bin'
            logging.info('call: %s', cmd)
            sp.Popen(cmd, shell=True, stdout=pl.stdout, stderr=sp.STDOUT).wait()
            os.symlink(
                path_suggest_dict + '/suggest_framework/suggest_scripts/suggest_builder/suggest_filters/data/compiled_data',
                './compiled_data')

        os.mkdir('./suggest_filters')
        for mask in ['', 'tr', 'kz', 'town']:
            dst_dir = './suggest_filters/bad_combinations' + ('.' if mask else '') + mask
            logging.info('create dir:%s', dst_dir)
            os.mkdir(dst_dir)
            for filename in ['exceptions.gzt.bin', 'interactions', 'markers.gzt.bin', 'objects.gzt.bin']:
                srcfile = './compiled_data/' + mask + ('.' if mask else '') + filename
                dstfile = dst_dir + '/' + filename
                logging.info('copy: %s->%s', srcfile, dstfile)
                shutil.copy(srcfile, dstfile)

        os.mkdir('./suggest_filters/words_filters')
        for fn in ['porno.lst', 'porno.nokb.lst', 'tr.porno.dic']:
            srcfile = path_suggest_dict + '/suggest_framework/suggest_scripts/suggest_builder/suggest_filters/data/' + fn
            dstdir = './suggest_filters/words_filters'
            logging.info('copy: %s->%s', srcfile, dstdir)
            shutil.copy(srcfile, dstdir)

        os.mkdir('./suggest_filters/regexp_filters')
        for fn in ['stop.pref',
                   'stop.nokb.pref',
                   'porno.pref',
                   'porno.except',
                   'porno.nokb.pref',
                   'tr.porno.pref',
                   'porno.pref.url',
                   'cycle.pref',
                   'stop.pref.particular',
                   'obscureurls.stop.pref',
                   'tr.ignore.pref',
                   'images.stop.pref',
                   'porno.pref.url']:
            srcfile = path_suggest_dict + '/suggest_framework/suggest_scripts/suggest_builder/suggest_filters/data/' + fn
            dstdir = './suggest_filters/regexp_filters'
            logging.info('copy: %s->%s', srcfile, dstdir)
            shutil.copy(srcfile, dstdir)
            fullname = dstdir + '/' + fn
            command = ['./bin/regexp_list_precom', '--input', fullname, '--output', fullname + '.json']
            run_process(command, log_prefix="regexp_list_precom", wait=True, shell=True)

        dstdir = './suggest_filters/fixlists'
        os.mkdir(dstdir)
        shutil.copy(path_suggest_dict + '/suggest_framework/suggest_scripts/suggest_builder/suggest_filters/data/static_ban_list.txt', dstdir)

        srcfile = path_suggest_dict + '/suggest_framework/suggest_scripts/suggest_builder/suggest_filters/config.xml'
        dstdir = './suggest_filters'
        shutil.copy(srcfile, dstdir)

        result_path = './result'
        os.mkdir(result_path)
        shutil.move('./suggest_filters', './result/suggest_filters')
        shutil.make_archive(result_path, 'tar', result_path)
        result_tar_path = './result.tar'
        result_tar_encrypted_path = './suggest_filters.tar.encrypted'
        fernet = Fernet(fernet_key.encode('utf-8'))
        with open(result_tar_path, 'rb') as f, open(result_tar_encrypted_path, 'wb') as of:
            of.write(fernet.encrypt(f.read()))

        resource = resource_types.SuggestFilters(self, "Encrypted suggest filters", result_tar_encrypted_path)
        data = sdk2.ResourceData(resource)
        data.ready()

        api = common.rest.Client()
        dict_name = 'suggest_filters'
        api.resource[resource.id].attribute(name="name", value=dict_name)
        api.resource[resource.id].attribute(name="ttl", value=999)
        api.resource[resource.id].attribute(name="autodeploy", value=self.Parameters.autodeploy)
        logging.info('resoruce %i', resource.id)

# end
