from sandbox import sdk2
from sandbox.sandboxsdk import environments

from dateutil.parser import parse
from dateutil.tz import tzutc

import datetime
import os
import json
import logging


class TableWatcher(sdk2.Task):
    """ Deletes old YT tables. """

    class Parameters(sdk2.Task.Parameters):
        yt_token_vault_name = sdk2.parameters.String(
            'Name of the vault record with YT token',
            name='vault_token_name',
            default='SUGGEST_YT_TOKEN'
        )
        """
        It is a python dict that may have all parameters from
        python3 datetime.timedelta constructor
        > import datetime
        > help(datetime.timedelta)
        """
        config_arcadia_url = sdk2.parameters.String(
            'Path to configuration file at Arcadia',
            name='config_arcadia_url',
            default='quality/trailer/suggest/base/configs/table_watcher.json',
            required=True
        )

    class Requirements(sdk2.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
        )

    def on_execute(self):
        import yt.wrapper as yt
        proxy = "hahn.yt.yandex.net"
        token = sdk2.Vault.data(self.Parameters.yt_token_vault_name)
        yt_client = yt.YtClient(proxy=proxy, token=token)

        logging.info("Tables to remove:")

        now = datetime.datetime.now(tz=tzutc())
        jsons = sdk2.svn.Arcadia.cat(sdk2.svn.Arcadia.trunk_url(self.Parameters.config_arcadia_url))
        data = json.loads(jsons)

        def dfs(dir_name, delta):
            for node in yt_client.list(dir_name):
                node_type = yt_client.get(os.path.join(dir_name, node + "&", "@type"))
                if node_type == "table":
                    tab_dt = parse(yt_client.get(os.path.join(dir_name, node, "@modification_time")))
                    if now.astimezone(tab_dt.tzinfo) - tab_dt > delta:
                        table_to_delete = os.path.join(dir_name, node)
                        logging.info("\t" + table_to_delete)
                        yt_client.remove(table_to_delete)
                elif node_type == "map_node":
                    dfs(os.path.join(dir_name, node), delta)

        for dir_name in data.keys():
            delta_args = data[dir_name]
            delta = datetime.timedelta(**delta_args)
            dfs(dir_name, delta)
