import os
import logging
import json
import yaml

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.suggest.watchdog import watchdog_tests
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class SuggestWatchdog(sdk2.Task):
    ''' Watchdog tests for suggest '''

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        arcadia_config_paths = sdk2.parameters.String('Comma separated config paths in Arcadia', default='')
        target_instance = sdk2.parameters.String('Target instance to test (with http(s)://)', default='')
        nanny_service = sdk2.parameters.String('Nanny service to test', default='')
        vault_user = sdk2.parameters.String('Sandbox vault user having nanny token and git ssh key', default='')
        nanny_token_name = sdk2.parameters.String('Nanny oauth token name from Sandbox vault', default='')
        git_ssh_key_name = sdk2.parameters.String('Git ssh key name from Sandbox vault', default='')
        git_repo_url = sdk2.parameters.String('Url of git repository with config', default='')
        git_branch = sdk2.parameters.String('Branch of git repository with config', default='')
        git_config_dir = sdk2.parameters.String('Relative path to configs directory in git repository', default='')
        allowed_persentage = sdk2.parameters.Integer('Persentage of successfully tested instances to success (without symbol %). For example, 50', default=50)
        print_errors_count = sdk2.parameters.Integer('Print errors count', default=10)
        tvm_src = sdk2.parameters.Integer('Source service id for tvm ticket', default=0)
        tvm_dst = sdk2.parameters.Integer('Destination service id for tvm ticket', default=0)
        tvm_secret_name = sdk2.parameters.String('Vault secret name with tvm secret', default='')

        with sdk2.parameters.RadioGroup('Hostname source for nanny instances') as hostname_source:
            hostname_source.values['container_hostname'] = hostname_source.Value(value='container_hostname')
            hostname_source.values['hostname'] = hostname_source.Value(value='hostname', default=True)

    def on_execute(self):
        # Svn export watchdog configs to local dir
        configs_dir = 'watchdog_configs'
        os.mkdir(configs_dir)

        if (self.Parameters.arcadia_config_paths):
            self.download_configs_from_arcadia(configs_dir)
        elif (self.Parameters.git_config_dir):
            self.download_configs_from_git(configs_dir)
        else:
            raise SandboxTaskFailureError('You have to set arcadia_config_paths or git_config_paths parameters')

        if (self.Parameters.print_errors_count):
            self.Context.remain_print_errors = self.Parameters.print_errors_count
            self.Context.some_errors = []

        if self.Parameters.target_instance:
            logging.info('Target instance to test: %s', self.Parameters.target_instance)
            if not self.run_tests(configs_dir, target_instance=self.Parameters.target_instance):
                raise SandboxTaskFailureError('Some tests for target instance {} have failed. See logs.'.format(self.Parameters.target_instance))

        elif self.Parameters.nanny_service:
            if not self.Parameters.vault_user or not self.Parameters.nanny_token_name:
                raise SandboxTaskFailureError('You have set nanny_service "{}" as a target for test. So set vault_user and nanny_token_name please too!'.format(self.Parameters.nanny_service))

            nanny_oauth_token = sdk2.Vault.data(self.Parameters.vault_user, self.Parameters.nanny_token_name)
            nanny_client = nanny.NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=nanny_oauth_token)

            logging.info('Target nanny service to test: %s', self.Parameters.nanny_service)
            service_instances = nanny_client.get_service_current_instances(self.Parameters.nanny_service)['result']
            if not len(service_instances):
                raise SandboxTaskFailureError('Nanny service {} does not contain instances!'.format(self.Parameters.nanny_service))

            logging.debug('service_instances: %s', json.dumps(service_instances))
            successful_instances_count = 0

            for service_instance in service_instances:
                service_instance_name = 'http://{}:{}'.format(service_instance[self.Parameters.hostname_source], service_instance['port'])
                logging.info('Target instance to test: %s', service_instance_name)
                if self.run_tests(configs_dir, target_instance=service_instance_name):
                    successful_instances_count += 1

            successful_percentage = successful_instances_count * 100 / len(service_instances)
            if successful_percentage < self.Parameters.allowed_persentage:
                raise SandboxTaskFailureError('Percentage of successful instances {}% less then allowed percentage {}%'.format(successful_percentage, self.Parameters.allowed_persentage))

        else:
            logging.info('target_instance or nanny_service does not set. So use target instances from configs')
            if not self.run_tests(configs_dir):
                raise SandboxTaskFailureError('Some tests have failed. See logs.')

    def run_tests(self, configs_dir, target_instance=''):
        status_ok = True
        for config_name in os.listdir(configs_dir):
            config_path = os.path.join(configs_dir, config_name)
            logging.info('Current watchdog config: %s', config_path)

            with open(config_path) as config_handler:
                try:
                    parsed_config = yaml.safe_load(config_handler)
                except yaml.YAMLError:
                    raise SandboxTaskFailureError('Can not parse yaml config: {}'.format(config_name))

                for test in parsed_config['tests']:
                    if isinstance(test, dict):
                        method_name = test.keys()[0]
                        method_args = test[method_name]
                    else:
                        method_name = test
                        method_args = {}

                    if target_instance:
                        real_target_instance = target_instance
                    elif 'instance' in method_args:
                        real_target_instance = method_args['instance']
                    elif 'default_instance' in parsed_config:
                        real_target_instance = parsed_config['default_instance']
                    else:
                        raise SandboxTaskFailureError('Set target instance please.')
                    method_args['instance'] = real_target_instance

                    if self.Parameters.tvm_src:
                        if not self.Parameters.tvm_dst or not self.Parameters.tvm_secret_name:
                            raise SandboxTaskFailureError(
                                'You have to set tvm_dst and tvm_secret_name')
                        method_args['tvm_ticket'] = self.get_tvm_ticket()

                    logging.debug('Run test %s for instance %s', method_name, real_target_instance)
                    logging.debug('Test args %s: ', json.dumps(method_args, ensure_ascii=False))
                    test_function = getattr(watchdog_tests, method_name)
                    try:
                        fail_msg = test_function(**method_args)
                    except Exception, e:
                        logging.error("Exception is happened: %s" % (e))
                        return False

                    logging.debug('Test is %s', 'OK' if not fail_msg else 'FAILED')

                    if fail_msg:
                        status_ok = False

                        if "remain_print_errors" in self.Context and self.Context.remain_print_errors > 0 and fail_msg not in self.Context.some_errors:
                            self.set_info(fail_msg)
                            self.Context.some_errors.append(fail_msg)
                            self.Context.remain_print_errors -= 1

        return status_ok

    def download_configs_from_arcadia(self, configs_dir):
        for arcadia_config_path in self.Parameters.arcadia_config_paths.split(','):
            arcadia_config_path = arcadia_config_path.strip()
            sdk2.svn.Arcadia.export(sdk2.svn.Arcadia.trunk_url(arcadia_config_path), configs_dir)

    def download_configs_from_git(self, configs_dir):
        if not self.Parameters.git_repo_url or not self.Parameters.git_branch or not self.Parameters.git_ssh_key_name:
            raise SandboxTaskFailureError("Git parameters should be presented")

        logging.info('Checkout git repository')
        git_checkout_path = 'git_repo'
        try:
            self.checkout_git_repo(git_checkout_path)
        except Exception, e:
            logging.info(str(e))
            self.checkout_git_repo(git_checkout_path)

        logging.info('Move config files to configs directory')
        config_dir_from_git = os.path.join(git_checkout_path, self.Parameters.git_config_dir)
        try:
            os.rename(config_dir_from_git, configs_dir)
        except Exception, e:
            raise SandboxTaskFailureError("Couldn't copy configs: %s" % (e))

    def checkout_git_repo(self, git_checkout_path):
        try:
            with sdk2.ssh.Key(self, self.Parameters.vault_user,
                              self.Parameters.git_ssh_key_name):
                sdk2.vcs.git.Git(self.Parameters.git_repo_url).clone(
                    git_checkout_path, self.Parameters.git_branch)
            return
        except Exception, e:
            raise SandboxTaskFailureError(
                "Couldn't checkout git repository: %s" % (e))

    def get_tvm_ticket(self):
        import ticket_parser2.api.v1 as tp2
        try:
            return tp2.TvmClient(tp2.TvmApiClientSettings(
                self_client_id=self.Parameters.tvm_src,
                self_secret=sdk2.Vault.data(self.Parameters.vault_user, self.Parameters.tvm_secret_name),
                dsts={"dst": self.Parameters.tvm_dst},
            )).get_service_ticket_for("dst")
        except Exception, e:
            raise SandboxTaskFailureError("Couldn't get tvm ticket: %s" % (e))
