# encoding: utf-8

"""Calculate product metrics for homework pushes and cards.

Uses binary resources created in: https://sandbox.yandex-team.ru/task/225946908/view (copy, check arcadia rev).
Don't forget to add a big ttl to the resource.
"""

import logging
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp


class HOME_WORK_METRICS_TOOL(sdk2.Resource):
    """
        HomeWorkMetricsTool binary to calculate homework metrics.
    """
    any_arch = True
    executable = True
    releasers = ['antonka', 'dpolukhin', 'zador', 'putrin']


class CalcHomeWorkMetrics(sdk2.Task):
    """Calculate homework product metrics for pushes and cards."""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 20 * 1024
        ram = 8 * 1024

    class Parameters(sdk2.Parameters):
        kill_timeout = 18000

        tables_date = sdk2.parameters.String(
            'Date to calculate metrics for.',
            default=''
        )
        do_metrika = sdk2.parameters.Bool(
            'Download metrika data.',
            default=True
        )
        do_push_stats = sdk2.parameters.Bool(
            'Download push events data.',
            default=True
        )
        do_subs = sdk2.parameters.Bool(
            'Calculate subscription metrics.',
            default=True
        )
        do_ctr = sdk2.parameters.Bool(
            'Calculate sent/delivered metrics.',
            default=True
        )
        do_geo_diff = sdk2.parameters.Bool(
            'Calculate geo difference metrics.',
            default=True
        )
        do_time_diff = sdk2.parameters.Bool(
            'Calculate time difference metrics.',
            default=True
        )
        do_join_publish = sdk2.parameters.Bool(
            'Join metrics and publish to StatFace.',
            default=True
        )

        push_logs_path = sdk2.parameters.String(
            'Sent pushes logs path.',
            default=''
        )
        raw_geo_path = sdk2.parameters.String(
            'Geo-positioning logs path.',
            default=''
        )
        hw_data_path = sdk2.parameters.String(
            'Homework data path.',
            default=''
        )

    def on_execute(self):
        _tables_date = self.Parameters.tables_date
        _dates = [""]
        if _tables_date:
            _dates = _tables_date.split()

        for _date in _dates:
            cmd_args = []

            if _date:
                cmd_args.extend(['--date', _date])
                logging.info('Processing date ' + _date)
            else:
                logging.info('Processing current date')

            # String parameters
            for key, value in (
                ['--push-logs-path', self.Parameters.push_logs_path],
                ['--raw-geo-path', self.Parameters.raw_geo_path],
                ['--hw-data-path', self.Parameters.hw_data_path],
            ):
                if value:
                    cmd_args.extend([key, value])

            # Boolean parameters (flags)
            for key, value in (
                ['--metrika', self.Parameters.do_metrika],
                ['--push-stats', self.Parameters.do_push_stats],
                ['--subs', self.Parameters.do_subs],
                ['--ctr', self.Parameters.do_ctr],
                ['--geo-diff', self.Parameters.do_geo_diff],
                ['--time-diff', self.Parameters.do_time_diff],
                ['--join-publish', self.Parameters.do_join_publish],
            ):
                if value:
                    cmd_args.append(key)

            homework_metrics_binary_resources = sdk2.Resource.find(
                resource_type=HOME_WORK_METRICS_TOOL,
                state='READY'
            ).limit(3)

            last_binary = sorted(list(homework_metrics_binary_resources), key=lambda x: x.created)[-1]
            last_binary_path = str(sdk2.ResourceData(last_binary).path)

            logging.info('Using HomeWorkMetricsTool binary created on: ' + last_binary.created.strftime('%Y-%m-%d'))
            logging.info('=======\n\n', last_binary_path, cmd_args, '=======\n\n')

            os.environ['YT_TOKEN'] = sdk2.Vault.data(self.owner, 'robot_sup_yt_token')
            os.environ['YQL_TOKEN'] = sdk2.Vault.data(self.owner, 'robot_sup_yql_token')
            os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'

            with sdk2.helpers.ProcessLog(self, logger="home_work_metrics") as pl:
                sp.check_call([last_binary_path] + cmd_args, stdout=pl.stdout, stderr=sp.STDOUT)
