import os
import re
import time
import logging

from sandbox import sdk2
from sandbox.sandboxsdk.environments import ArcEnvironment
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.helpers import ProcessLog
from sandbox.projects.common.arcadia import sdk as arcadia_sdk

TEMPLATE = '''"{type}": {{
    "limit": {limit},
    "state": "{state}",
    "request": ${{{project}.{preset_name}}} {{
        "receivers": {{
            "apps": [
{apps}
            ]
        }}
    }},
    "issue": "{issue}",
    "responsible": "{responsible}",
    "abc_service": "{abc_service}",
    "category": "{category}",
    "topic": "{topic}",
    "important": {important},
    "send_to_bell": {send_to_bell}
}},'''


def count_spaces(config, pos):
    pos -= 1
    space_prefix = ''
    while pos >= 0 and config[pos] != '\n' and config[pos].isspace():
        space_prefix += config[pos]
        pos -= 1
    return space_prefix


def tab_string(s, space_prefix):
    res = ''
    for line in s.split('\n'):
        res += space_prefix + line + '\n'
    return res[:-1]


class ConfigAutoShipper(sdk2.Task):
    class Parameters(sdk2.Parameters):
        project = sdk2.parameters.String("project")
        pushtype = sdk2.parameters.String("pushtype")
        limit = sdk2.parameters.Integer("limit")
        state = sdk2.parameters.String("state")
        apps = sdk2.parameters.String("apps")
        preset_name = sdk2.parameters.String("preset_name")
        pushlaunch = sdk2.parameters.String("pushlaunch")
        responsible = sdk2.parameters.String("responsible")
        task = sdk2.parameters.String("task")
        abc_service = sdk2.parameters.String("abc_service")
        category = sdk2.parameters.String("category")
        topic = sdk2.parameters.String("topic")
        important = sdk2.parameters.Bool("important")
        send_to_bell = sdk2.parameters.Bool("send_to_bell")

    """ Update config/* """

    def on_execute(self):
        with arcadia_sdk.mount_arc_path('arcadia-arc:/#trunk') as arcadia_root:
            arc_token = sdk2.Vault.data('SUP', 'ARC_TOKEN')
            env = {
                'ARC_TOKEN': arc_token,
                'USER': 'robot-sup'
            }

            path = 'sup/push/src/main/resources/config'
            project_path = os.path.join(arcadia_root, path)
            commit_msg = self.Parameters.task or self.Parameters.pushlaunch
            summary = self.Parameters.pushlaunch

            with ProcessLog(self, logger='arc') as arc_log:
                arc = ArcEnvironment().prepare()
                branch_name = 'sandbox/{}/{}'.format(self.Parameters.pushlaunch, str(time.time()))

                filename = self.Parameters.project + '.json'
                with open(os.path.join(project_path, filename), 'r') as f:
                    config = f.read()
                pushtypes = str(self.Parameters.pushtype).split(',')
                pushtypes = [t.strip() for t in pushtypes]
                pushtypes = [t for t in pushtypes if t != '*' and t]
                for pushtype in pushtypes:
                    config = self.edit_config(config, pushtype)
                with open(os.path.join(project_path, filename), 'w') as f:
                    f.write(config)
                logging.info(config)

                retcode = subprocess.Popen('{} co -b {}'.format(arc, branch_name), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=project_path, env=env).wait()
                assert retcode == 0, retcode
                retcode = subprocess.Popen('{} add {}'.format(arc, filename), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=project_path, env=env).wait()
                assert retcode == 0, retcode
                retcode = subprocess.Popen('{} ci -m "{}"'.format(arc, commit_msg.replace('"', '\\"')), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=project_path, env=env).wait()
                assert retcode == 0, retcode
                retcode = subprocess.Popen('{} pr create --push --publish -m "{}"'.format(arc, summary), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=project_path, env=env).wait()
                assert retcode == 0, retcode

    @staticmethod
    def generate_apps(apps):
        filled = ''
        app_template = 16 * ' ' + '${{"*".preset.receivers.apps."{app}"}},'
        for app in apps:
            filled += app_template.format(app=app) + '\n'
        return filled[:-1]

    @staticmethod
    def remove_pushtype(config, pushtype):
        splitted = config.split('\n')
        type_start = -1
        type_end = -1
        for idx, line in enumerate(splitted):
            match = re.search('"{pushtype}".*:.*{{'.format(pushtype=pushtype), line, re.MULTILINE)
            if match:
                type_start = idx
                break
        if type_start == -1:
            return config

        balance = 0
        for idx, line in enumerate(splitted[type_start:]):
            for c in line:
                if c == '{':
                    balance += 1
                elif c == '}':
                    balance -= 1
            if balance == 0:
                type_end = idx + type_start
                break
        splitted = splitted[0: type_start] + splitted[type_end + 1:]
        return '\n'.join(splitted)

    def edit_config(self, config, pushtype):
        config = self.remove_pushtype(config, pushtype)
        filled_template = TEMPLATE.format(
            type=pushtype,
            limit=self.Parameters.limit,
            state=self.Parameters.state,
            project=str(self.Parameters.project),
            preset_name=str(self.Parameters.preset_name),
            apps=self.generate_apps(str(self.Parameters.apps).split(',')),
            issue=str(self.Parameters.pushlaunch),
            responsible=str(self.Parameters.responsible),
            abc_service=self.Parameters.abc_service,
            category=self.Parameters.category,
            topic=self.Parameters.topic,
            important=str(self.Parameters.important).lower(),
            send_to_bell=str(self.Parameters.send_to_bell).lower()
        )
        match = re.search('"\\*"[^\\.]', config, re.MULTILINE)
        start, _ = match.span()
        space_prefix = count_spaces(config, start)
        start -= len(space_prefix)
        before, after = config[:start], config[start:]
        before += tab_string(filled_template, space_prefix)
        return before + '\n' + after
