import os
import logging
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp


class GEO_TRACKING_TOOL(sdk2.Resource):
    """
        geo_tracking resource type
    """
    any_arch = True
    executable = True
    releasers = ['riariorovere', 'jon2205', 'zador', 'robot-sup']


class GeoTracking(sdk2.Task):
    """ GeoTracking in AppMetrikaLog per day. """

    class Requirements(sdk2.Requirements):
        disk_space = 20 * 1024
        ram = 8 * 1024
        environments = [environments.PipEnvironment('python-step-client')]

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 21600

        # custom parameters
        source_path1 = sdk2.parameters.String('Source path (metrika-mobile-log)', required=True,
                                              default='//logs/metrika-mobile-log/1d')
        source_path2 = sdk2.parameters.String('Source path (appmetrica-location-log)', required=True,
                                              default='//statbox/appmetrica-location-log')
        destination = sdk2.parameters.String('Destination path', required=True, default='//home/sup/geo_tracking')
        current_table = sdk2.parameters.String('Set table name if not need last (or array of tables separated by comma)', default=None)
        limit = sdk2.parameters.String('Limit source table records (for fast processing)', default=None)
        step_event_type = sdk2.parameters.String('STEP sup_table_ready event type field', required=False)

    def on_execute(self):
        cmd_args = [self.Parameters.source_path1, self.Parameters.source_path2, self.Parameters.destination]
        if self.Parameters.current_table:
            cmd_args.extend(['-tbl', self.Parameters.current_table])
        if self.Parameters.limit:
            cmd_args.extend(['-lmt', self.Parameters.limit])

        os.environ['YQL_TOKEN'] = sdk2.Vault.data(self.owner, 'robot_sup_yql_token')
        os.environ['YT_TOKEN'] = sdk2.Vault.data(self.owner, 'robot_sup_yt_token')

        geo_tracking_binary_resources = sdk2.Resource.find(
            resource_type=GEO_TRACKING_TOOL,
            state='READY'
        ).limit(3)

        last_binary = sorted(list(geo_tracking_binary_resources), key=lambda x: x.created)[-1]
        last_binary_path = str(sdk2.ResourceData(last_binary).path)

        logging.info('Using GeoTracking binary created on: ' + last_binary.created.strftime('%Y-%m-%d'))
        logging.info('=======\n\n', last_binary_path, cmd_args, '=======\n\n')

        with sdk2.helpers.ProcessLog(self, logger="geo_tracking") as pl:
            sp.check_call([last_binary_path] + cmd_args, stdout=pl.stdout, stderr=sp.STDOUT)

        from step import Client
        if self.Parameters.step_event_type:
            step_client = Client(host='https://step.sandbox.yandex-team.ru', timeout=(1, 1))
            event_id = step_client.create_event(
                name='sup_table_ready',
                params={'type': self.Parameters.step_event_type},
                sandbox_task_id=self.id)
            logging.info('STEP event: {}'.format(event_id))

        self.set_info('Task is OK')
        logging.info('Task is OK')
