# encoding: utf-8

"""Calculate user geo activities and routes for home/work points; send cards and pushes.

Uses binary resources created in: https://sandbox.yandex-team.ru/task/147139866/view (copy, check arcadia rev).
Don't forget to add a big ttl to the resource.
"""

import logging
import os

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp


class HOME_WORK_TOOL(sdk2.Resource):
    """
        HomeWorkTool binary to get home/work user data and schedule, upload cards and send pushes.
    """
    any_arch = True
    executable = True
    releasers = ['antonka', 'dpolukhin', 'zador', 'putrin']


class HomeWork(sdk2.Task):
    """Download home/work data, send cards and pushes."""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 20 * 1024
        ram = 8 * 1024

    class Parameters(sdk2.Parameters):
        kill_timeout = 83000

        activity_data = sdk2.parameters.Bool(
            'Calculate user activity.',
            default=True
        )
        build_cards = sdk2.parameters.Bool(
            'Create appsearch card data.',
            default=True
        )
        build_pushes = sdk2.parameters.Bool(
            'Create appsearch push data.',
            default=True
        )

        with build_pushes.value[True]:
            with sdk2.parameters.CheckGroup("Push apps select") as push_apps_select:
                push_apps_select.values.search_app = push_apps_select.Value("Create pushes for search_app.")
                push_apps_select.values.yaphone = push_apps_select.Value("Create pushes for yaphone.")
                push_apps_select.values.navi = push_apps_select.Value("Create pushes for navi.")

        send_cards = sdk2.parameters.Bool(
            'Send cards to bass.',
            default=True
        )
        reduce_log = sdk2.parameters.Bool(
            'Join log records for all stages.',
            default=True
        )
        update_holidays = sdk2.parameters.Bool(
            'Update holiday calendar.',
            default=True
        )
        send_pushes = sdk2.parameters.Bool(
            'Send pushes to users based on immediate conditions.',
            default=False
        )
        loud_pushes = sdk2.parameters.Bool(
            'Send pushes with sound.',
            default=False
        )
        show_button = sdk2.parameters.Bool(
            'Send pushes with button.',
            default=False
        )
        tables_date = sdk2.parameters.String(
            'Date for home/work table.',
            default=''
        )
        update_staff = sdk2.parameters.Bool(
            'Update staff user base.',
            default=False
        )
        root_path = sdk2.parameters.String(
            'Path to root folder.',
            default=''
        )
        users_path = sdk2.parameters.String(
            'Users table path.',
            default=''
        )
        times_path = sdk2.parameters.String(
            'Times table path.',
            default=''
        )
        resources = sdk2.parameters.String(
            'Path to library resources on YT.',
            default=''
        )
        sup_table = sdk2.parameters.String(
            'SUP users base on YT.',
            default=''
        )
        cards_filter = sdk2.parameters.String(
            'Table to filter cards before send.',
            default=''
        )
        exp_name = sdk2.parameters.String(
            'Experiment folder.',
            default=''
        )
        exp_id = sdk2.parameters.String(
            'exp_id field of the pushes.',
            default=''
        )

        with sdk2.parameters.String('Binary', multiline=True) as task_binary:
            task_binary.values.Custom = None
            task_binary.values.FetchLatest = task_binary.Value(default=True)

        with task_binary.value['Custom']:
            home_work_bin = sdk2.parameters.Resource(
                'Resource with bin to use',
                resource_type=HOME_WORK_TOOL,
                required=True
            )

        with sdk2.parameters.Output:
            used_binary = sdk2.parameters.Resource(
                'Resource with bin used in process',
                resource_type=HOME_WORK_TOOL,
            )

    def _find_bin_resource(self):
        task_binary = self.Parameters.task_binary

        if task_binary == 'Custom':
            return self.Parameters.home_work_bin

        homework_binary_resources = list(
            sdk2.Resource.find(
                resource_type=HOME_WORK_TOOL,
                state='READY'
            ).limit(3)
        )

        sorted_resources = sorted(homework_binary_resources, key=lambda x: x.created, reverse=True)
        if sorted_resources:
            return sorted_resources[0]

    def _get_push_apps_cli_args(self):
        if self.Parameters.push_apps_select:
            return ['--push-devices'] + self.Parameters.push_apps_select

        return []

    def on_execute(self):
        cmd_args = []

        # String parameters
        for key, value in (
            ['--date', self.Parameters.tables_date],
            ['--root-path', self.Parameters.root_path],
            ['--users-path', self.Parameters.users_path],
            ['--times-path', self.Parameters.times_path],
            ['--resources', self.Parameters.resources],
            ['--sup-base', self.Parameters.sup_table],
            ['--cards-filter', self.Parameters.cards_filter],
            ['--exp-name', self.Parameters.exp_name],
            ['--exp-id', self.Parameters.exp_id],
        ):
            if value:
                cmd_args.extend([key, value])

        # Boolean parameters (flags)
        for key, value in (
            ['--activity', self.Parameters.activity_data],
            ['--build-cards', self.Parameters.build_cards],
            ['--build-pushes', self.Parameters.build_pushes],
            ['--send-cards', self.Parameters.send_cards],
            ['--send-pushes', self.Parameters.send_pushes],
            ['--update-staff', self.Parameters.update_staff],
            ['--reduce-log', self.Parameters.reduce_log],
            ['--update-holidays', self.Parameters.update_holidays],
            ['--loud-push', self.Parameters.loud_pushes],
            ['--hide-button', not self.Parameters.show_button],
        ):
            if value:
                cmd_args.append(key)

        cmd_args.extend(self._get_push_apps_cli_args())

        bin_resource = self._find_bin_resource()

        if not bin_resource:
            raise common.errors.TaskFailure("No binaries found!")

        self.Parameters.used_binary = bin_resource

        last_binary_path = str(sdk2.ResourceData(self.Parameters.used_binary).path)
        logging.info('Using HomeWorkTool binary created on: ' + self.Parameters.used_binary.created.strftime('%Y-%m-%d'))
        logging.info('=======\n\n', last_binary_path, cmd_args, '=======\n\n')

        os.environ['BASS_TOKEN'] = sdk2.Vault.data('antonka_bass_token')
        os.environ['STAFF_TOKEN'] = sdk2.Vault.data('robot_sup_staff_token')
        os.environ['PUSH_TOKEN'] = sdk2.Vault.data('robot_sup_sup_token')
        os.environ['YT_TOKEN'] = sdk2.Vault.data('robot_sup_yt_token')
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'

        with sdk2.helpers.ProcessLog(self, logger="home_work") as pl:
            sp.check_call([last_binary_path] + cmd_args, stdout=pl.stdout, stderr=sp.STDOUT)
