# encoding: utf-8

"""Calculate metrics (time deviation) for homework pushes to test methods.

Uses binary resources created in: https://sandbox.yandex-team.ru/task/<task_id>/view (copy, check arcadia rev).
Don't forget to add a big ttl to the resource.
"""

import logging
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp


class HOME_WORK_TEST_TOOL(sdk2.Resource):
    """
        HomeWorkTestTool binary to test new methods and calculate time deviation.
    """
    any_arch = True
    executable = True
    releasers = ['antonka', 'dpolukhin', 'zador', 'putrin']


class HomeWorkTest(sdk2.Task):
    """Calculate metrics (time deviation) for homework pushes to test methods."""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 20 * 1024
        ram = 8 * 1024

    class Parameters(sdk2.Parameters):
        kill_timeout = 18000

        tables_date = sdk2.parameters.String(
            'Date/Dates to calculate metrics for.',
            default=''
        )

        push_logs_path = sdk2.parameters.String(
            'Sent pushes logs path.',
            default=''
        )
        raw_geo_path = sdk2.parameters.String(
            'Geo-positioning logs path.',
            default=''
        )
        hw_data_path = sdk2.parameters.String(
            'Homework data path.',
            default=''
        )

    def on_execute(self):
        cmd_args = []

        _date = self.Parameters.tables_date
        if _date:
            cmd_args.extend(['--date', _date])

        # String parameters
        for key, value in (
            ['--push-logs-path', self.Parameters.push_logs_path],
            ['--raw-geo-path', self.Parameters.raw_geo_path],
            ['--hw-data-path', self.Parameters.hw_data_path],
        ):
            if value:
                cmd_args.extend([key, value])

        homework_metrics_binary_resources = sdk2.Resource.find(
            resource_type=HOME_WORK_TEST_TOOL,
            state='READY'
        ).limit(3)

        last_binary = sorted(list(homework_metrics_binary_resources), key=lambda x: x.created)[-1]
        last_binary_path = str(sdk2.ResourceData(last_binary).path)

        logging.info('Using HomeWorkTestTool binary created on: ' + last_binary.created.strftime('%Y-%m-%d'))
        logging.info('=======\n\n', last_binary_path, cmd_args, '=======\n\n')

        os.environ['YT_TOKEN'] = sdk2.Vault.data('antonka', 'YT_TOKEN')
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'

        with sdk2.helpers.ProcessLog(self, logger="home_work_test") as pl:
            sp.check_call([last_binary_path] + cmd_args, stdout=pl.stdout, stderr=sp.STDOUT)
