import os
import logging
import datetime
from sandbox import common
from sandbox import sdk2
from sandbox.sandboxsdk import environments


class HomeWorkYql(sdk2.Task):
    """ Run YQL using SUP databse. """

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('requests'),
            environments.PipEnvironment('python-step-client'),
        ]

    class Parameters(sdk2.Task.Parameters):
        # custom parameters
        yql_query = sdk2.parameters.String('YQL Query', required=True, multiline=True)
        src_path = sdk2.parameters.String('Source path', required=True,
                                          default='//home/datasync/fulldump/profile/addresses/common_addresses/current')
        dst_path = sdk2.parameters.String('Destination path', required=True,
                                          default='//home/search-functionality/sup/home_work/hw')
        sup_path = sdk2.parameters.String('Sup users path', required=True,
                                          default='//home/search-functionality/sup/backup/sup-prod.n.yandex-team.ru')
        yql_var = sdk2.parameters.String('YQL VAR', required=False, multiline=True)
        yql_token_name = sdk2.parameters.String('YQL token vault name', required=True, default='robot_sup_yql_token')
        yt_token_name = sdk2.parameters.String('YT token vault name VAR', required=True, default='robot_sup_yt_token')
        step_event_type = sdk2.parameters.String('STEP sup_table_ready event type field', required=False)

    def on_execute(self):
        yql_token = sdk2.Vault.data(self.owner, self.Parameters.yql_token_name)
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_name)
        yt_proxy = 'hahn.yt.yandex.net'

        if not yt_token or not yql_token:
            raise common.errors.TaskError('Empty token param')

        import yt.wrapper as yt
        yt.config['proxy']['url'] = yt_proxy
        yt.config['token'] = yt_token

        path_to_users = None
        sup_path = yt.list(self.Parameters.sup_path)
        for path in reversed(sup_path):
            temp_path = os.path.join(self.Parameters.sup_path, path, 'sup_users')
            if yt.exists(temp_path):
                path_to_users = temp_path
                break
        if not path_to_users:
            raise common.errors.TaskError('Empty user tables in sup users path')
        query = self.expand(path_to_users)
        logging.info(query)

        from yql.api.v1.client import YqlClient
        yql_client = YqlClient(
            token=yql_token,
        )
        request = yql_client.query(query)
        request.run()
        request.get_results(wait=True)
        if not request.is_success:
            error_description = '\n'.join([str(err) for err in request.errors])
            logging.error(error_description)

        from step import Client
        if self.Parameters.step_event_type:
            step_client = Client(host='https://step.sandbox.yandex-team.ru', timeout=(1, 1))
            event_id = step_client.create_event(
                name='sup_table_ready',
                params={'type': self.Parameters.step_event_type},
                sandbox_task_id=self.id)
            logging.info('STEP event: {}'.format(event_id))

    def expand(self, sup_path):
        query = self.Parameters.yql_query
        if not query:
            raise common.errors.TaskError('Empty query param')
        today = datetime.datetime.now()
        yesterday = today - datetime.timedelta(days=1)
        query_placeholders = {'%TODAY%': today.strftime('%Y-%m-%d'), '%YEAR%': today.strftime('%Y'),
                              '%MONTH%': today.strftime('%m'), '%DAY%': today.strftime('%d'),
                              '%YESTERDAY%': yesterday.strftime('%Y-%m-%d'),
                              '%YESTERDAY_YEAR%': yesterday.strftime('%Y'),
                              '%YESTERDAY_MONTH%': yesterday.strftime('%m'),
                              '%YESTERDAY_DAY%': yesterday.strftime('%d'), '%SUP%': sup_path}
        yql_var = self.Parameters.yql_var
        if yql_var:
            for key, value in query_placeholders.iteritems():
                yql_var = yql_var.replace(key, value)
            query_placeholders['%VAR%'] = yql_var

        dst = self.Parameters.dst_path
        for key, value in query_placeholders.iteritems():
            dst = dst.replace(key, value)
        query_placeholders['%DST%'] = dst

        src = self.Parameters.src_path
        for key, value in query_placeholders.iteritems():
            src = src.replace(key, value)
        query_placeholders['%SRC%'] = src

        for key, value in query_placeholders.iteritems():
            query = query.replace(key, value)
        return query
