import os
import shutil
from distutils.version import LooseVersion
import logging

from sandbox import sdk2

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.environments import SandboxMavenEnvironment
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.resource_types import VPS_JDK_DIST
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox.sandboxsdk.process import run_process


class VpsJdkEnviroment(SandboxEnvironment):
    resource_type = VPS_JDK_DIST

    name = 'jdk'

    @property
    def find_resource(self):
        version = LooseVersion(self.version)
        resources = [
            resource
            for release in channel.sandbox.list_releases(self.resource_type, ctm.OSFamily.LINUX, limit=100)
            for resource in release.resources
        ]
        for resource in resources:
            if LooseVersion(str(resource.attributes['version'])) >= version:
                self.resource_id = resource.id
                return channel.task.sync_resource(resource)
        raise common.errors.SandboxEnvironmentError("Couldn't find resource for version {}".format(self.version))

    def prepare(self):
        path = self.find_resource
        if not os.path.exists(path):
            shutil.copytree(self.get_environment_resource(), path)
        os.environ['PATH'] = os.path.join(path, 'bin') + ':' + os.environ['PATH']
        os.environ['JAVA_HOME'] = os.path.join(path)
        os.environ['LD_LIBRARY_PATH'] = ':'.join(
            filter(None, [os.path.join(path, 'lib'), os.environ.get('LD_LIBRARY_PATH')])
        )
        os.environ['XIVA_SEND_TOKEN'] = 'xiva-send-token'
        os.environ['XIVA_SUBSCRIBE_TOKEN'] = 'xiva-subscribe-token'
        os.environ['TVM_SECRET'] = 'tvm-secret'


class SvnVcsRoot:
    def __init__(self, url, location):
        self.url = url
        self.location = location

    def obtain(self):
        if os.path.exists(self.location):
            shutil.rmtree(self.location)
        run_process(['svn', 'checkout', self.url, self.location], log_prefix='svn')

        rev = (
            run_process(
                ['svn', 'info', '|', 'awk', "'/^Last Changed Rev:/ {print $4}'"],
                'svn',
                work_dir=self.location,
                outs_to_pipe=True,
                shell=True,
            )
            .stdout.read()
            .strip()
        )
        self.vcs_input_ref = self.url
        self.vcs_output_ref = "svn://" + rev

        return self


class ArcVcsRoot:
    def __init__(self, url):
        self.url = url

    def obtain(self):
        url = 'arcadia-arc:/#{}'.format(self.url)

        self.arcadia = arcadiasdk.mount_arc_path(url)
        mount_path = self.arcadia.__enter__()
        self.location = '{}sup/push'.format(mount_path)

        rev = run_process(
            ['arc', 'rev-parse', 'HEAD'],
            'arc',
            work_dir=self.location,
            outs_to_pipe=True
        ).stdout.read().strip()

        self.vcs_input_ref = "arc://{}".format(self.url)
        self.vcs_output_ref = "arc://{}".format(rev)


class PushDist(sdk2.Resource):
    """
        Jars for push
    """
    vcs = sdk2.resource.Attributes.String("VCS url")
    releasable = True
    any_arch = False
    executable = False
    releasers = ['robot-sup', 'SUP']


# noinspection PyAbstractClass
class PushBuild(release_integration.ReleaseToNannyAndYaDeployTask2, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        # system parameter overrides
        kill_timeout = 30 * 60  # Task timeout in seconds
        enable_yav = True
        # custom parameters
        svn = sdk2.parameters.String("Svn reference", required=False)
        arc = sdk2.parameters.String("Arc reference", required=False)
        env_type = sdk2.parameters.StrictString(
            "Environment", default='testing', regexp='production|testing|development'
        )
        mvn_cmd = sdk2.parameters.String("Addditional Maven command line parameters", required=False)
        yt_token_secret = sdk2.parameters.String("YT token secret", default='sec-01daxj1zv0cd2ryq1dvn89zx6d')
        yp_token_secret = sdk2.parameters.YavSecret("YP token secret (must contain token in yp-token field)", default='sec-01g8g4tej806az45880xj7xq25')
        skip_tests = sdk2.parameters.Bool("Production is down (aka skip tests)", default=False)
        suspend_before_upload = sdk2.parameters.Bool("[debug] Suspend before creating resource", default=False)

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 8 * 1024
        ram = 16 * 1024
        cores = 6
        environments = [SandboxMavenEnvironment('3.2.2'), VpsJdkEnviroment('1.8.0')]

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    def on_execute(self):
        self.obtain()
        self.build()
        self.publish()

    def obtain(self):
        if self.Parameters.svn:
            self.vcs_root = SvnVcsRoot(self.Parameters.svn, str(self.path('push')))
        else:
            assert self.Parameters.arc
            self.vcs_root = ArcVcsRoot(self.Parameters.arc)

        self.vcs_root.obtain()

        self.location = str(self.vcs_root.location)

    def build(self):
        env = os.environ.copy()

        env['YT_TOKEN'] = sdk2.yav.Secret(self.Parameters.yt_token_secret).data()['secret']
        params = ['-Druntime.environment=%s' % (self.Parameters.env_type), '-Dsandbox.id=%s' % (self.id,), '-Pdist']
        if self.Parameters.skip_tests:
            params.append('-Dmaven.test.skip')
        goals = ['package']
        run_process(
            ['mvn', '-B'] + params + self.Parameters.mvn_cmd.split() + goals,
            'mvn',
            work_dir=self.location,
            environment=env,
        )

    def publish(self):
        # when using arc, sandbox can not find built resources in arcadia fuse
        # so as a workaround we copy artifacts to intermediate location
        built_path = self.location + "/target/releases/pushd"
        dist_path = str(self.path("./pushd"))
        if self.Parameters.suspend_before_upload:
            self.suspend()
        shutil.copytree(built_path, dist_path)
        logging.info("publishing artifact from %s", dist_path)
        logging.debug(os.listdir(dist_path))
        in_ref = self.vcs_root.vcs_input_ref
        out_ref = self.vcs_root.vcs_output_ref
        res = PushDist(self, "SUP push jar from {} ({})".format(in_ref, out_ref), dist_path)
        sdk2.ResourceData(res).ready()

    def get_yp_oauth_token(self):
        return self.Parameters.yp_token_secret.data()['yp-token']


    def on_release(self, additional_parameters):
        release_integration.ReleaseToNannyAndYaDeployTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)

    def postprocess(self):
        pass
