# -*- coding: utf-8 -*-
"""Get failed push request statuses and send to StatFace."""

import os
import json
import logging

from datetime import date, timedelta

from sandbox.sandboxsdk import environments
from sandbox import sdk2
from sandbox.sdk2.helpers.process import subprocess as sp


class PushFailedToSendStats(sdk2.Task):
    """Get failed push request statuses and send to StatFace."""
    SUBPROCESS_RUNNER_NAME = 'query_runner'

    class Requirements(sdk2.Task.Requirements):
        environments = [
             environments.PipEnvironment('python-statface-client', version='0.87.0', use_wheel=True)
         ]

    @staticmethod
    def create_statface_config(creds):
        username = creds['username']
        password = creds['password']
        home = os.path.expanduser('~')
        statbox_dir = home + '/.statbox'
        statbox_config = statbox_dir + '/statface_auth.yaml'

        os.mkdir(statbox_dir, 0755)

        with os.fdopen(os.open(statbox_config, os.O_WRONLY | os.O_CREAT, 0664), 'w') as f:
            f.write('---\n')
            f.write('username: %s\n' % username)
            f.write('password: %s\n' % password)

    @staticmethod
    def upload_to_statface():
        import statface_client
        day = (date.today() - timedelta(days=1)).strftime('%Y-%m-%d')
        cli = statface_client.StatfaceClient(host=statface_client.STATFACE_PRODUCTION)
        r = cli.get_report('Yandex/Others/push_fail_stats')
        r.upload_yt_data(
            scale='d',
            cluster='hahn',
            table_path='{}/{}/{}'.format(
                '//home/search-functionality/sup/push_stats',
                day,
                'push_failed_to_send'
            ),
        )

    def on_execute(self):
        logging.info("Creating virtualenv")
        os.environ['YQL_TOKEN'] = sdk2.Vault.data('robot_sup_yql_token')
        statface_credentials = json.loads(sdk2.Vault.data('STATFACE_CREDENTIALS'))
        self.create_statface_config(statface_credentials)

        with environments.VirtualEnvironment() as venv:
            environments.PipEnvironment('pip', venv=venv, version=environments.PipEnvironment.PIP_VERSION).prepare()
            environments.PipEnvironment('yql', venv=venv, version='1.2.49', use_wheel=True).prepare()

            script_path = os.path.join(
                os.path.dirname(os.path.realpath(__file__)),
                '{}.py'.format(self.SUBPROCESS_RUNNER_NAME)
            )
            process_description = [
                venv.executable, script_path,
            ]

            logging.info("Launching process %s", process_description)
            with sdk2.helpers.ProcessLog(self, logger=self.SUBPROCESS_RUNNER_NAME) as pl:
                p = sp.Popen(process_description, stdout=pl.stdout, stderr=pl.stderr)
                code = p.wait()

            if code != 0:
                raise RuntimeError("Process exited with code {}. Check OUT and ERR log for logger {}".format(
                    code,
                    self.SUBPROCESS_RUNNER_NAME
                ))
            else:
                self.upload_to_statface()

        logging.info("Successfully finished process execution")
