import logging
import os
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn
import shutil

import requests

from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.nanny import nanny


# TODO: rename to PushProjectsConfig
class PUSH_PROJECTS_CONFIG_2(sdk2.Resource):
    """
        applications-priorities.json
    """
    releasable = True
    any_arch = False
    executable = False
    releasers = ['SUP']


class SupAutoReleaseProjectsConfig(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600

        rev = sdk2.parameters.String('revision', required=True)
        check_only = sdk2.parameters.Bool('Check only (do not deploy config)')

        suspend_before_publish_resource = sdk2.parameters.Bool('[Debug] Suspend before publishing resource')

        notifications = [
            sdk2.Notification(
                statuses=[
                    ctt.Status.FAILURE,
                    ctt.Status.EXCEPTION,
                    ctt.Status.TIMEOUT
                ],
                recipients=["sup-dev@yandex-team.ru"],
                transport=ctn.Transport.EMAIL
            )
        ]

    class Requirements(sdk2.Requirements):
        ram = 512
        disk_space = 3522
        cores = 0.5

    def on_execute(self):
        self.descr = self.Parameters.rev
        logging.info('checkout config')
        with arcadiasdk.mount_arc_path("arcadia-arc:#" + self.Parameters.rev) as aarcadia:
            config_location = '{}sup/push/src/main/resources/config'.format(aarcadia)
            os.remove(os.path.join(config_location, 'ya.make'))
            os.remove(os.path.join(config_location, 'a.yaml'))
            temp_dir = str(self.path('config'))
            shutil.copytree(config_location, temp_dir)
            logging.info('Publishing files at %s', temp_dir)
            resource = PUSH_PROJECTS_CONFIG_2(
                self,
                'config',
                temp_dir
            )
            if self.Parameters.suspend_before_publish_resource:
                self.suspend()
            sdk2.ResourceData(resource).ready()
        self.validate_config(resource)

    def on_success(self, prev_status):
        logging.info("task succeed")
        sdk2.Task.on_success(self, prev_status)
        if self.Parameters.check_only:
            logging.info("check_only is enabled, skipping deployment")
            return
        for status in [ctt.ReleaseStatus.UNSTABLE, ctt.ReleaseStatus.PRESTABLE, ctt.ReleaseStatus.STABLE]:
            self.on_release(dict(
                releaser=self.author,
                release_status=status,
                release_subject="SupAutoReleaseProjectsConfig",
                email_notifications=dict(to=[], cc=[]),
                release_comments="SupAutoReleaseProjectsConfig",
            ))
        logging.info("task release complete")

    def on_release(self, additional_parameters):
        super(SupAutoReleaseProjectsConfig, self).on_release(additional_parameters)

    def validate_config(self, resource):
        logging.info("start validate new config")
        resource_data = sdk2.ResourceData(resource)  # synchronizing resource data on disk
        resource_path = str(resource_data.path)
        config_data_pattern = '"{0}": {1}'
        config_data = config_data_pattern.format("*", open(os.path.join(resource_path, 'base-applications-priorities.json'), 'r').read())
        with open(os.path.join(resource_path, 'project-types.config'), 'r') as f:
            for line in f:
                line = line.replace("\n", "")
                config_data += config_data_pattern.format(line, open(os.path.join(resource_path, line + '.json'), 'r').read())

        host = self.get_host()
        response = requests.post(host, data=config_data)
        if response.status_code != 200:
            raise Exception("Validation fail: " + response.text)

    def get_host(self):
        return 'http://sup.yandex.net/admin/validate-config'
