import logging
import os
from sandbox import common
import requests
import time
import urlparse

from sandbox import sdk2

SAVE_COLLECTION = '/data'


class SupLoadCollection(sdk2.Task):
    class Parameters(sdk2.Parameters):
        yt_path = sdk2.parameters.String(
            'YT source table',
            required=True)

        sup_host = sdk2.parameters.String(
            'SUP host',
            default='push-beta.n.yandex-team.ru',
            required=True)

        collection = sdk2.parameters.String(
            'Target collection',
            required=True)

        column = sdk2.parameters.String(
            'Value column name',
            required=True)

        append = sdk2.parameters.Bool(
            'Append data',
            default=False
        )

        wait = sdk2.parameters.Bool(
            'Sequential run',
            default=False
        )

    def on_execute(self):
        parts = self.Parameters.yt_path.split(':')
        table = parts.pop(0) if parts else None
        chunk_size = int(parts.pop(0)) if parts else None
        limit = int(parts.pop(0)) if parts else None
        wait = self.Parameters.wait
        mode = 'append' if self.Parameters.append else 'load'
        collection = self.Parameters.collection
        url = urlparse.urlunsplit(
            ('http', self.Parameters.sup_host, os.path.join(SAVE_COLLECTION, mode, collection), None, None))
        logging.info('Requesting SUP url: %s', url)
        response = requests.post(url,
                                 json={
                                     'path': table,
                                     'chunk_size': chunk_size,
                                     'limit': limit,
                                     'index_column': self.Parameters.column
                                 },
                                 headers={'Content-Type': 'application/json'})
        if response.status_code != 200:
            raise common.errors.TaskError('SUP API Error (%d):\n%s' % (response.status_code, response.text))
        json = response.json()
        job_id = json.get('jobExecutionId')
        if job_id is None:
            raise common.errors.TaskError('SUP API Error (%d):\n%s' % (response.status_code, response.text))
        logging.info('Start loading of %s using %s into %s', (table, job_id, collection))
        if not wait:
            return job_id

        front_host = response.headers['X-Yandex-Front']
        if front_host:
            host = front_host
        else:
            raise common.errors.TaskError('Failed to find the task executor')

        front_port = response.headers['X-Yandex-Front-Port']
        if front_port:
            host += ':' + front_port
        url = urlparse.urlunsplit(('http', host, os.path.join(SAVE_COLLECTION, 'status', str(job_id)), None, None))
        try:
            while True:
                try:
                    response = requests.get(url, headers={'Content-Type': 'application/json'})
                except requests.exceptions.ReadTimeout:
                    continue

                if response.status_code == 504:
                    continue

                if response.status_code != 200:
                    raise common.errors.TaskError(
                        'SUP API Error (%d):\n%s' % (response.status_code, response.text))
                if not wait:
                    break
                json = response.json()
                status = json.get('status')
                if status == 'COMPLETED':
                    return json.get('jobExecutionId')
                elif status == 'FAILED':
                    raise common.errors.TaskError('Load failed (%s):\n%s' % (url, json.get('exitDescription')))
                time.sleep(5.0)
        except KeyboardInterrupt:
            return None
        logging.info('Load is finished for %s using %s into %s', (table, job_id, collection))
        return job_id
