import logging
import os
import time
import urlparse
import requests

import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.task import SandboxTask

ADMIN_DEDUP = '/admin/dedup'


class ShardCountParam(sdk_parameters.SandboxIntegerParameter):
    name = 'shard_count'
    description = 'Shard count'
    default_value = 4
    required = True


class WaitEnd(sdk_parameters.SandboxBoolParameter):
    name = 'wait_end'
    description = 'Wait end of task'
    default_value = True
    required = True


class SupHostParam(sdk_parameters.SandboxStringParameter):
    name = 'sup_host'
    description = 'Host'
    required = True
    default_value = 'push-beta.n.yandex-team.ru'


class SupRemoveOutdatedTask(SandboxTask):
    type = 'SUP_REMOVE_OUTDATED'
    input_parameters = (ShardCountParam, WaitEnd, SupHostParam)

    def on_execute(self):
        host = self.ctx[SupHostParam.name]
        wait = self.ctx[WaitEnd.name]
        url = urlparse.urlunsplit(('http', host, ADMIN_DEDUP, None, None))
        response = requests.post(url,
                                 json={},
                                 headers={'Content-Type': 'application/json'})
        if response.status_code != 200:
            raise errors.SandboxTaskFailureError('SUP API Error (%d):\n%s' % (response.status_code, response.text))
        json = response.json()
        job_id = json.get('jobExecutionId')
        if job_id is None:
            raise errors.SandboxTaskFailureError('SUP API Error (%d):\n%s' % (response.status_code, response.text))
        logging.info('Starting deduplication using %s ', job_id)

        if not wait:
            return job_id
        front_host = response.headers['X-Yandex-Front']
        if front_host:
            host = front_host
        front_port = response.headers['X-Yandex-Front-Port']
        if front_port:
            host += ':' + front_port
        url = urlparse.urlunsplit(('http', host, os.path.join(ADMIN_DEDUP, str(job_id)), None, None))
        logging.info(url)
        try:
            while True:
                try:
                    response = requests.get(url, headers={'Content-Type': 'application/json'})
                except requests.exceptions.ReadTimeout:
                    continue

                if response.status_code == 504:
                    continue

                if response.status_code != 200:
                    raise errors.SandboxTaskFailureError(
                        'SUP API Error (%d):\n%s' % (response.status_code, response.text))
                if not wait:
                    break
                json = response.json()
                status = json.get('status')
                if status == 'COMPLETED':
                    return json.get('jobExecutionId')
                elif status == 'FAILED':
                    raise errors.SandboxTaskFailureError(
                        'Deduplication failed (%s):\n%s' % (url, json.get('exitDescription')))
                time.sleep(5.0)
        except KeyboardInterrupt:
            return None
        logging.info('Finished deduplication using %s', job_id)
        return job_id


__Task__ = SupRemoveOutdatedTask
